/*
  This file is part of CDO. CDO is a collection of Operators to manipulate and analyse Climate model Data.

  Author: Uwe Schulzweida

*/

#include <cdi.h>

#include "cdo_options.h"
#include "process_int.h"
#include "param_conversion.h"
#include "pmlist.h"
#include "cdo_default_values.h"

static void
set_query_parameter(const KVList &kvlist, CdiQuery *query)
{
  for (const auto &kv : kvlist)
    {
      const auto &key = kv.key;
      if (kv.nvalues < 1) cdo_abort("Missing value for parameter key >%s<!", key);

      // clang-format off
      if (key == "name")
        {
          const int numQueryNames = kv.nvalues;
          std::vector<char*> queryNames(numQueryNames);
          for (int i = 0; i < numQueryNames; ++i)
            {
              queryNames[i] = (char*)kv.values[i].c_str();
            }
          cdiQuerySetNames(query, numQueryNames, queryNames.data());
        }
      else cdo_abort("Invalid parameter key >%s<!", key);
      // clang-format on
    }
}

void *
Query(void *process)
{
  cdo_initialize(process);

  cdo_operator_add("query", 0, 0, "query entries");

  const auto dataIsUnchanged = data_is_unchanged();

  const auto operatorID = cdo_operator_id();

  operator_input_arg(cdo_operator_enter(operatorID));

  const auto natts = cdo_operator_argc();
  if (natts == 0) cdo_abort("Parameter missing!");

  PMList pmlist;
  KVList kvlist;
  kvlist.name = "QUERY";
  if (kvlist.parse_arguments(natts, cdo_get_oper_argv()) != 0) cdo_abort("Parse error!");
  if (Options::cdoVerbose) kvlist.print();

  auto pkvlist = &kvlist;
  if (natts == 1)
    {
      KeyValues &kv = kvlist.front();
      if (kv.key == "FILE")
        {
          if (Options::cdoVerbose) cdo_print("Reading query from: %s", kv.values[0]);
          auto filename = parameter_to_word(kv.values[0].c_str());
          auto fp = fopen(filename, "r");
          if (fp == nullptr) cdo_abort("Open failed on: %s\n", filename);
          pmlist.read_namelist(fp, filename);
          pkvlist = &pmlist.front();
          fclose(fp);
          if (Options::cdoVerbose) pkvlist->print();
        }
    }

  CdiQuery *query = cdiQueryCreate();
  set_query_parameter(*pkvlist, query);
  cdiQueryPrint(query);

  const auto streamID1 = streamOpenReadQuery(cdo_get_stream_name(0), query);
  //const auto streamID1 = streamOpenRead(cdo_get_stream_name(0));
  cdiQueryDelete(query);

  const auto filetype = streamInqFiletype(streamID1);
  if (CdoDefault::FileType == CDI_UNDEFID) CdoDefault::FileType = filetype;

  const auto vlistID1 = streamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdo_open_write(1);
  cdo_def_vlist(streamID2, vlistID2);

  Field field;

  VarList varList1;
  varListInit(varList1, vlistID1);

  int tsID = 0;
  while (true)
    {
      const auto nrecs = streamInqTimestep(streamID1, tsID);
      if (nrecs == 0) break;

      cdo_taxis_copy_timestep(taxisID2, taxisID1);
      cdo_def_timestep(streamID2, tsID);

      printf("tsID = %d\n", tsID);
      for (int recID = 0; recID < nrecs; ++recID)
        {
          int varID, levelID;
          streamInqRecord(streamID1, &varID, &levelID);
          cdo_def_record(streamID2, varID, levelID);
          /*
          if (dataIsUnchanged)
            {
              streamCopyRecord(streamID2, streamID1);
            }
          else
          */
            {
              field.init(varList1[varID]);
              if (field.memType == MemType::Float)
                streamReadRecordF(streamID1, field.vec_f.data(), &field.nmiss);
              else
                streamReadRecord(streamID1, field.vec_d.data(), &field.nmiss);
              cdo_write_record(streamID2, field);
            }
        }

      tsID++;
    }

  streamClose(streamID1);
  cdo_stream_close(streamID2);

  cdo_finish();

  return nullptr;
}
