(in-package "DM")

(include-book "../sym")


;; For n >= 3, consider a regular n-gon with vertices labeled 0, 1, ..., n-1 in clockwise order.  The dihedral group,
;; (dihedral n), is the subgroup of (sym n) of order 2n consisting of the permutations of these vertices that may be
;; realized by rigid motion of the n-gon in 3-dimensional space.  Each of these is either a rotation or a reflection
;; through some axis of symmetry.  The group is generated by 2 elements, the first of which is the basic clockwise
;; rotation by 2*pi/n radians, which has order n:

(defund rot (n) (append (cdr (ninit n)) (list 0)))

(defthmd sublistp-cdr-ninit
    (implies (posp n)
             (sublistp (cdr (ninit n)) (ninit n))))

(defthmd rot-sublist-ninit
    (implies (posp n)
             (sublistp (rot n) (ninit n)))
  :hints (("Goal" :in-theory (enable rot)
                  :use (sublistp-cdr-ninit
                        (:instance sublistp-append
                                   (l (cdr (ninit n))) (m (list 0)) (g (ninit
                                                                        n)))))))
(defthm dlistp-rot
    (implies (posp n)
             (dlistp (rot n)))
  :hints (("Goal" :in-theory (enable rot)
                  :use ((:instance dlistp-append
                                   (l (cdr (ninit n))) (m (list 0)))
                        (:instance dlistp-remove1
                                   (x (car (ninit n))) (l (ninit n)))
                        (:instance disjointp-symmetric
                                   (l (list 0)) (m (cdr (ninit n))))))))

(defthm permp-rot
  (implies (posp n)
           (member-equal (rot n) (slist n)))
  :hints (("Goal" :in-theory (enable rot)
                  :use (sublistp-cdr-ninit
                        (:instance member-perm-slist (x (rot n)))
                        (:instance len-remove1-equal
                                   (x (car (ninit n))) (l (ninit n)))
                        dlistp-rot))))

(defthm nth-cdr
    (implies (and (natp i) (true-listp l) (< (1+ i) (len l)))
             (equal (nth i (cdr l)) (nth (1+ i) l))))

(defthm nth-rot
    (implies (and (posp n) (natp i) (< i n))
             (equal (nth i (rot n))
                    (mod (1+ i) n)))
  :hints (("Goal" :in-theory (enable rot)
                  :use ((:instance len-remove1-equal
                                        (x (car (ninit n))) (l (ninit n)))))
          ("Subgoal 2" :use ((:instance rtl::mod-does-nothing
                                        (m (1+ i)) (n n))))))

(defthm len-rot-ninit
    (implies (posp n)
             (equal (len (rot n)) (len (ninit n))))
  :hints (("Goal" :in-theory (enable rot))))

(defthm index-rot
    (implies (and (posp n) (natp i) (< i n))
             (equal (index i (rot n))
                    (mod (1- i) n)))
  :hints (("Goal" :cases ((= 0 i) (< 0 i))
                  :use ((:instance ind-nth
                            (i (mod (1- i) n)) (l (rot n)))))))

(defun index-recurse (i)
  (if (zp i)
      i
      (index-recurse (1- i))))

(defthmd power-rot-nth
    (implies (and (posp n) (natp i) (< i n) (natp j))
             (equal (nth i (power (rot n) j (sym n)))
                    (mod (+ i j) n)))
  :hints (("Goal" :in-theory (enable e)
                  :induct (index-recurse j))
          ("Subgoal *1/2" :use ((:instance sym-op-rewrite
                                           (x (rot n))
                                           (y (power (rot n) (1- j) (sym n))))
                                (:instance rtl::mod-sum
                                           (a 1) (b (+ -1 i j)) (n n))
                                (:instance power-in-g
                                           (a (rot n)) (n (1- j)) (g (sym n)))))))

(defthmd power-rot
    (implies (posp n)
             (equal (power (rot n) n (sym n)) (e (sym n))))
  :hints (("Goal" :in-theory (enable e power-rot-nth)
                  :use ((:instance nth-diff-perm
                                   (x (power (rot n) n (sym n)))
                                   (y (ninit n)))
                        (:instance power-in-g
                                   (a (rot n)) (g (sym n)))))))

(defthmd nth-diff-not-equal
    (implies (and (natp k) (< k (len l))
                  (not (= (nth k l) (nth k m))))
             (not (equal l m))))

(defthmd ord-aux-rot-neq
    (implies (and (posp n) (posp i) (< i n))
             (not (equal (e (sym n))
                         (power (rot n) i (sym n)))))
  :hints (("Goal" :in-theory (enable power e)
                  :use ((:instance nth-diff-not-equal
                                   (k 0) (l (ninit n))
                                   (m (power (rot n) i (sym n))))
                        (:instance power-rot-nth
                                   (i 0) (j i))))))

;; for assisting ord-rot
(encapsulate ()
  (local (defthm fact<=
             (implies (posp n) (<= n (fact n)))))
  (local (defthm fact<
             (implies (and (< i n) (<= n (fact n)))
                        (< i (fact n)))))
  (local (defun ord-aux-rot-induct (i n)
           (declare (xargs :measure (nfix (- n i))))
           (if (zp (- n i))
               n
               (ord-aux-rot-induct (1+ i) n))))
  (local (defthm ord-aux-rot
             (implies (and (posp n) (posp i) (< i n))
                      (equal (ord-aux (rot n) i (sym n))
                             (ord-aux (rot n) (+ 1 i) (sym n))))
           :hints (("Goal" :in-theory (enable order-sym ord-aux-rot-neq)))))

  (defthmd ord-aux-rot<n
      (implies (and (posp n) (posp i) (< i n))
               (equal (ord-aux (rot n) i (sym n)) n))
    :hints (("Goal" :induct (ord-aux-rot-induct i n))
             ("Subgoal *1/2" :use ((:instance power-rot (n (1+ i))))))))

(defthm ord-rot
    (implies (posp n)
             (equal (ord (rot n) (sym n))
                    n))
  :hints (("Goal" :in-theory (enable ord)
                  :use ((:instance ord-aux-rot<n (i 1))))))

;; The second element is the reflection about the axis of symmetry that passes through the center of the n-gon and 
;; the midpoint of the side connecting vertices 0 and n - 1, which has order 2:

(defund ref (n) (reverse (ninit n)))

(defthmd sublistp-rev
    (implies (true-listp l)
             (sublistp (reverse l) l)))

(defthm dlistp-rev
    (implies (dlistp l)
             (dlistp (reverse l)))
  :hints (("Subgoal *1/2" :use ((:instance sublistp-rev (l (cdr l)))
                                (:instance member-sublist
                                   (x (car l)) (l (acl2::rev (cdr l))) (m (cdr l)))))))

(defthm permp-ref
  (implies (posp n)
           (member-equal (ref n) (slist n)))
  :hints (("Goal" :in-theory (enable ref)
                  :use ((:instance member-perm-slist (x (ref n)))
                        (:instance sublistp-rev (l (ninit n)))
                        (:instance dlistp-rev (l (ninit n)))))))


(defund nth-rev-induct (i l)
  (if (not (and (consp l) (natp i)))
      ()
      (if (equal i 0)
          (list i l)
          (nth-rev-induct (1- i) (cdr l)))))

(defthmd nth-rev
    (implies (and (true-listp l) (natp i) (< i (len l)))
             (equal (nth (- (1- (len l)) i) (reverse l)) (nth i l)))
  :hints (("Goal" :in-theory (enable nth-rev-induct)
                  :induct (nth-rev-induct i l))))

(defthm nth-ref
    (implies (and (posp n) (natp i) (< i n))
             (equal (nth i (ref n))
                    (- (1- n) i)))
  :hints (("Goal" :in-theory (enable ref)
                  :use ((:instance nth-rev
                                   (i (- (1- n) i)) (l (ninit n)))))))
(defthm len-ref
    (implies (posp n)
             (equal (len (ref n)) n))
  :hints (("Goal" :in-theory (enable ref))))

(defthm nth-inv-perm-ref
    (implies (and (posp n) (natp i) (< i n))
             (equal (nth i (inv-perm (ref n) n))
                    (nth i (ref n))))
  :hints (("Goal" :in-theory (enable dlistp-perm)
	          :use ((:instance nth-ref (i (- (1- n) i)))
                        (:instance ind-nth
                                   (i (- (1- n) i))
                                   (l (ref n)))))))

(defthmd inv-perm-ref
    (implies (posp n)
             (equal (inv-perm (ref n) n) (ref n)))
  :hints (("Goal" :use ((:instance nth-diff-perm
                                   (x (ref n))
                                   (y (inv-perm (ref n) n)))))))

(defthmd power-ref
    (implies (and (natp n) (> n 1))
             (equal (power (ref n) 2 (sym n)) (e (sym n))))
  :hints (("Goal" :use ((:instance comp-perm-inv-perm (x (ref n))))
                  :in-theory (enable e inv-perm-ref))))

(defthm ord-ref
    (implies (and (natp n) (> n 1))
             (equal (ord (ref n) (sym n))
                    2))
  :hints (("Goal" :in-theory (enable ord ord-aux power-ref))
          ("Subgoal 2" :in-theory (enable e)
                       :use ((:instance nth-diff-not-equal
                                        (k 0) 
                                        (l (ref n))
                                        (m (ninit n)))))
          ("Subgoal 1" :in-theory (enable order-sym))))

;; The element list of (dihedral n) consists of the n powers of (rot n) together with the composition of
;; (ref n) with each of these powers:

(defun map-comp-perm (p l n)
  ;; Compose a permutation p with each member of a list l of permutations:
  (if (consp l)
      (cons (comp-perm p (car l) n)
            (map-comp-perm p (cdr l) n))
    ()))

(defun delts (n)
  (append (powers (rot n) (sym n))
          (map-comp-perm (ref n) (powers (rot n) (sym n)) n)))

;; Computing the group operation on (dihedral n) is easy once we have the following:
(defthmd nth-comp-perm-ref-rot
    (implies (and (natp n) (natp i) (>= n 3) (< i n))
             (equal (nth i (comp-perm (ref n) (rot n) n))
                    (mod (- (- n i) 2) n)))
  :hints (("Goal" :use ((:instance nth-comp-perm (k i)
                                   (x (ref n)) (y (rot n)))
                        (:instance nth-ref (i (mod (1+ i) n)))
                        nth-rot)
                  :in-theory (disable nth-ref nth-rot))))

(defthmd nth-comp-perm-inv-rot-ref
    (implies (and (natp n) (natp i) (>= n 3) (< i n))
             (equal (nth i (comp-perm (inv-perm (rot n) n) (ref n) n))
                    (mod (- (- n i) 2) n)))
  :hints (("Goal" :use ((:instance nth-comp-perm (k i)
                                   (x (inv-perm (rot n) n))
                                   (y (ref n)))))))

(defthm comp-perm-ref-rot
  (implies (and (natp n) (>= n 3))
           (equal (comp-perm (ref n) (rot n) n)
                  (comp-perm (inv-perm (rot n) n) (ref n) n)))
  :hints (("Goal" :in-theory (enable nth-diff
                                     nth-comp-perm-ref-rot
                                     nth-comp-perm-inv-rot-ref)
                  :use ((:instance nth-diff-perm
                                   (x (comp-perm (ref n) (rot n) n))
                                   (y (comp-perm (inv-perm (rot n) n) (ref n)
                                                 n)))))))
(in-theory (disable comp-perm-ref-rot))
(encapsulate
 ()
 (local (defthmd eq-power-comp-perm
	    (implies (and (posp n) (natp i) (natp j) (< i n) (< j n)
		          (equal (power (rot n) i (sym n))
			         (comp-perm (ref n) (power (rot n) j (sym n)) n)))
		     (equal (ref n) (power (rot n) (mod (+ n (- j) i) n) (sym n))))
	  :hints (("Goal" :use ((:instance power+ (a (rot n)) (g (sym n)) (n (+ i (- j) n)) (m j))
			        (:instance power-mod (a (rot n)) (g (sym n)) (n (+ n i)))
		          	(:instance right-cancel
					   (a (power (rot n) j (sym n)))
					   (x (power (rot n) (+ i (- j) n) (sym n)))
				      	   (y (ref n)) (g (sym n)))
				(:instance power-in-g (a (rot n)) (n (+ i (- j) n)) (g (sym n)))
				(:instance power-in-g (a (rot n)) (n j) (g (sym n)))
				(:instance power-mod (a (rot n)) (g (sym n)) (n (+ i (- j) n))))))))

 (local (defthmd not-eq-ref-power
	    (implies (and (natp n) (>= n 3) (natp i) (< i n))
		     (not (equal (ref n) (power (rot n) i (sym n)))))
	  :hints (("Goal" :use ((:instance nth-diff-not-equal
					   (k 0) (l (ref n)) (m (power (rot n) i (sym n))))
				(:instance power-rot-nth (i 0) (j i))))
	          ("Goal'5'" :use ((:instance nth-diff-not-equal
				              (k (1- n)) (l (ref n)) (m (power (rot n) i (sym n))))
				   (:instance power-rot-nth (i (1- n)) (j (1- n))))))))

 (defthmd member-powers-rot
     (implies (and (posp n) (member-equal p (powers (rot n) (sym n))))
	      (let ((i (index p (powers (rot n) (sym n)))))
		(and (< i n) (equal (power (rot n) i (sym n)) p))))
   :hints (("Goal" :use ((:instance ind<len (x p) (l (powers (rot n) (sym n))))
			 (:instance nth-ind (x p) (l (powers (rot n) (sym n))))
			 (:instance member-powers (g (sym n))
				    (a (rot n)) (n (index p (powers (rot n) (sym n)))))))))

 (local (defthmd len-map-comp-perm (equal (len (map-comp-perm p l n)) (len l))))

 (local (defthmd eq-extract-map-comp-perm-nth
	    (implies (and (posp n) (natp i) (< i (len l)))
		     (equal (nth i (map-comp-perm p l n))
			    (comp-perm p (nth i l) n)))
	  :hints (("Goal" :in-theory (enable comp-perm)))))
 
 (defthmd member-map-comp-perm
     (implies (and (posp n)
		   (member-equal p (map-comp-perm (ref n) (powers (rot n) (sym n)) n)))
	      (let ((i (index p (map-comp-perm (ref n) (powers (rot n) (sym n)) n))))
		(and (< i n) (equal (comp-perm (ref n) (power (rot n) i (sym n)) n) p))))
   :hints (("Goal" :use ((:instance ind<len (x p) (l (map-comp-perm (ref n) (powers (rot n) (sym n)) n)))
			 (:instance len-map-comp-perm (p (ref n)) (l (powers (rot n) (sym n))))))
	   ("Subgoal 1" :use ((:instance eq-extract-map-comp-perm-nth
					 (i (index p (map-comp-perm (ref n) (powers (rot n) (sym n)) n)))
					 (p (ref n)) (l (powers (rot n) (sym n))))
			      (:instance nth-ind (x p) (l (map-comp-perm (ref n) (powers (rot n) (sym n)) n)))
			      (:instance member-powers (a (rot n)) (g (sym n))
					 (n (index p (map-comp-perm (ref n) (powers (rot n) (sym n)) n))))))))

 (defthmd delts-intersect-n
     (implies (and (member-equal p (powers (rot n) (sym n)))
   		   (member-equal p (map-comp-perm (ref n) (powers (rot n) (sym n)) n))
		   (posp n))
	      (< n 3))
   :hints (("Goal" :use (member-powers-rot
		 	 member-map-comp-perm
		 	 (:instance eq-power-comp-perm
				    (i (index p (powers (rot n) (sym n))))
				    (j (index p (map-comp-perm (ref n)
							       (powers (rot n) (sym n)) n))))
			 (:instance not-eq-ref-power
				    (i (mod (+ n (index p (powers (rot n) (sym n)))
					       (- (index p (map-comp-perm (ref n)
									  (powers (rot n) (sym n)) n))))
					    n)))))))

 (defthmd delts-no-intersect
     (implies (and (natp n) (>= n 3) (member-equal p (powers (rot n) (sym n))))
	      (not (member-equal p (map-comp-perm (ref n) (powers (rot n) (sym n)) n))))
   :hints (("Goal" :use (member-powers-rot
			 delts-intersect-n))))

 (defthmd delts-disjoint
     (implies (and (natp n) (>= n 3))
	      (disjointp (powers (rot n) (sym n))
			 (map-comp-perm (ref n) (powers (rot n) (sym n)) n)))
   :hints (("Goal" :use ((:instance delts-no-intersect
				    (p (intersect-witness (powers (rot n) (sym n))
							  (map-comp-perm
							   (ref n) (powers (rot n) (sym n)) n))))
			 (:instance intersect-witness-member (l (powers (rot n) (sym n)))
				    (m (map-comp-perm (ref n) (powers (rot n) (sym n)) n))))))))


(encapsulate
 ()
 (local (defthmd not-member-map-comp-perm
	    (implies (and (posp n) (sublistp l (slist n)) (member-equal p (slist n))
			  (not (member-equal x l)) (member-equal x (slist n)))
		     (not (member-equal (comp-perm p x n) (map-comp-perm p l n))))
	  :hints (("Subgoal *1/3" :use ((:instance left-cancel (g (sym n)) (a p) (y (car l))))))))


 (local (defthmd dlistp-map-comp-perm
	    (implies (and (posp n) (dlistp l) (sublistp l (slist n))
			  (member-equal p (slist n)))
		     (dlistp (map-comp-perm p l n)))
	  :hints (("Subgoal *1/3" :use ((:instance not-member-map-comp-perm (x (car l)) (l (cdr l))))))))

 (defthmd dlistp-map-comp-perm-ref-rot
     (implies (posp n)
	      (dlistp (map-comp-perm (ref n) (powers (rot n) (sym n)) n)))
   :hints (("Goal" :use ((:instance dlistp-map-comp-perm (p (ref n))
				    (l (powers (rot n) (sym n))))
			 (:instance sublistp-powers (a (rot n)) (g (sym n))))))))

(defthm dlistp-delts
  (implies (and (natp n) (>= n 3))
	   (dlistp (delts n)))
  :hints (("Goal" :use ((:instance dlistp-append
				   (l (powers (rot n) (sym n)))
				   (m (map-comp-perm (ref n) (powers (rot n) (sym n)) n)))
			delts-disjoint
			dlistp-map-comp-perm-ref-rot))))

;;   `(implies ,cond (sublistp ,elts (elts ,grp)))
(defthmd sublistp-map-comp-perm
    (implies (and (posp n) (member-equal p (slist n))
                  (sublistp l (slist n)))
             (sublistp (map-comp-perm p l n) (slist n))))

(defthm sublistp-delts-sym
    (implies (posp n)
             (sublistp (delts n) (slist n)))
  :hints (("Goal" :use ((:instance sublistp-append
                                   (l (powers (rot n) (sym n)))
                                   (m (map-comp-perm (ref n)
                                                     (powers (rot n) (sym n))
                                                     n))
                                   (g (slist n)))
                        (:instance sublistp-powers
                                   (a (rot n)) (g (sym n)))
                        (:instance sublistp-map-comp-perm
                                   (p (ref n)) (l (powers (rot n) (sym n))))))))

;;   `(implies ,cond (consp ,elts))
(defthm consp-delts
    (implies (and (natp n) (>= n 3))
             (consp (delts n))))

;;   `(implies ,cond (equal (car ,elts) (e ,grp)))
(defthm delts-elts-identity
    (implies (and (natp n) (>= n 3))
             (equal (car (delts n)) (e (sym n)))))


; we'll represent an element in the form of comp-perm ref and power of rot, or power of rot
(defthmd member-powers-rot
    (implies (and (posp n) (member-equal p (powers (rot n) (sym n))))
	     (let ((i (index p (powers (rot n) (sym n)))))
	       (and (< i n) (equal (power (rot n) i (sym n)) p))))
  :hints (("Goal" :use ((:instance ind<len (x p) (l (powers (rot n) (sym n))))
			(:instance nth-ind (x p) (l (powers (rot n) (sym n))))
			(:instance member-powers (g (sym n))
				   (a (rot n)) (n (index p (powers (rot n) (sym n)))))))))

(defthmd member-map-comp-perm
    (implies (and (posp n)
		  (member-equal p (map-comp-perm (ref n) (powers (rot n) (sym n)) n)))
	     (let ((i (index p (map-comp-perm (ref n) (powers (rot n) (sym n)) n))))
	       (and (< i n) (equal (comp-perm (ref n) (power (rot n) i (sym n)) n) p))))
  :hints (("Goal" :use ((:instance ind<len (x p) (l (map-comp-perm (ref n) (powers (rot n) (sym n)) n)))
			(:instance len-map-comp-perm (p (ref n)) (l (powers (rot n) (sym n))))))
	  ("Subgoal 1" :use ((:instance eq-extract-map-comp-perm-nth
					(i (index p (map-comp-perm (ref n) (powers (rot n) (sym n)) n)))
					(p (ref n)) (l (powers (rot n) (sym n))))
			     (:instance nth-ind (x p)
					(l (map-comp-perm (ref n) (powers (rot n) (sym n)) n)))
			     (:instance member-powers (a (rot n)) (g (sym n))
					(n (index p (map-comp-perm (ref n) (powers (rot n) (sym n)) n))))))))

;; helper lemmas for closure
(defthmd power+-sym
    (implies (and (posp n) (member-equal x (slist n)) (natp i) (natp j))
	     (equal (comp-perm (power x i (sym n)) (power x j (sym n)) n)
		    (power x (+ i j) (sym n))))
  :hints (("Goal" :use ((:instance power+ (a x) (n i) (m j) (g (sym n)))
			(:instance power-in-g (a x) (n i) (g (sym n)))
			(:instance power-in-g (a x) (n j) (g (sym n)))
			(:instance sym-op-rewrite (x (power x i (sym n)))
				   (y (power x j (sym n))))))))

(defthmd power-comp-perm-assoc
    (implies (and (posp n) (member-equal x (slist n)) (member-equal y (slist n)) (natp i))
	     (equal (comp-perm (comp-perm x (power y i (sym n)) n) (power y 1 (sym n)) n)
		    (comp-perm x (power y (1+ i) (sym n)) n)))
  :hints (("Goal" :use ((:instance sym-assoc (y (power y i (sym n)))
				   (z (power y 1 (sym n))))
			(:instance power+-sym (x y) (j 1))
			(:instance power-in-g (a y) (n i) (g (sym n)))
			(:instance power-in-g (a y) (n 1) (g (sym n)))))))

(defthmd mod-power
    (implies (and (posp n) (natp i))
	     (equal (power (rot n) (+ 1 (mod (+ -1 i) n)) (sym n))
		    (power (rot n) (mod i n) (sym n))))
  :hints (("Goal" :cases ((equal (mod (+ 1 (mod (+ -1 i) n)) n)
				 (+ 1 (mod (+ -1 i) n))))
		  :in-theory (disable power)
		  :use ((:instance rtl::mod-sum (a 1)
				   (b (+ -1 i)) (n n))))
	  ("Subgoal 2.1.1" :use ((:instance power-rot))
			   :in-theory (enable power))))

(defthmd power-mod-minus
    (implies (and (posp n) (< i n) (natp i))
	     (equal (power (rot n) (- n (mod (- n i) n)) (sym n))
		    (power (rot n) i (sym n))))
  :hints (("Goal" :cases ((equal i 0)))))


;; more lemmas (about simplifying group operations) for proving closure
(defthmd comp-perm-simplify-right
    (implies (and (natp n) (>= n 3) (natp i) (< i n))
	     (equal (comp-perm (ref n) (power (rot n) i (sym n)) n)
		    (comp-perm (power (rot n) (- n i) (sym n)) (ref n) n)))
  :hints (("Subgoal *1/5" :use ((:instance sym-assoc (x (ref n)) (y (rot n))
					   (z (power (rot n) (- i 1) (sym n))))
				(:instance power-in-g (a (rot n)) (n (1- i)) (g (sym n)))
				comp-perm-ref-rot
				(:instance sym-op-rewrite (x (rot n))
					   (y (power (rot n) (1- i) (sym n))))))
	  ("Subgoal *1/5'5'" :in-theory (disable sym-assoc)
			     :use ((:instance sym-assoc (x (inv-perm (rot n) n)) (y (ref n))
					      (z (power (rot n) (1- i) (sym n))))))
	  ("Subgoal *1/5'9'" :in-theory (disable sym-assoc)
			     :use ((:instance sym-op-rewrite (x (rot n))
					      (y (power (rot n) (- n i) (sym n))))
				   (:instance power-in-g (a (rot n)) (n (- n i)) (g (sym n)))
				   (:instance sym-assoc (x (inv-perm (rot n) n))
					      (y (comp-perm (rot n)
							    (power (rot n) (- n i) (sym n))
							    n))
					      (z (ref n)))
				   (:instance sym-assoc (x (inv-perm (rot n) n))
					      (y (rot n)) (z (power (rot n) (- n i) (sym n))))))
	  ("Subgoal *1/1" :in-theory (enable e)
			  :use ((:instance group-right-identity (x (ref n)) (g (sym n)))
				(:instance group-left-identity (x (ref n)) (g (sym n)))
				power-rot))))

(defthmd comp-perm-simplify
    (implies (and (natp n) (>= n 3) (natp i) (natp j) (< i n) (< j n))
	     (equal (comp-perm (power (rot n) i (sym n))
			       (comp-perm (ref n) (power (rot n) j (sym n)) n) n)
		    (comp-perm (ref n)
			       (power (rot n) (mod (+ n (- i) j) n) (sym n)) n)))
  :hints (("Goal" :in-theory (enable index-recurse)
		  :induct (index-recurse j))
	  ("Subgoal *1/2" :in-theory (disable power)
			  :use ((:instance power+ (a (rot n)) (g (sym n)) (n (1- j)) (m 1))
				(:instance sym-assoc
					   (x (ref n)) (y (power (rot n) (1- j) (sym n)))
					   (z (power (rot n) 1 (sym n))))
				(:instance sym-op-rewrite (x (power (rot n) (1- j) (sym n)))
					   (y (power (rot n) 1 (sym n))))
				(:instance sym-assoc (x (power (rot n) i (sym n)))
					   (y (ref n)) (z (power (rot n) (1- j) (sym n))))
				(:instance power-in-g (a (rot n)) (n (1- j)) (g (sym n)))
				(:instance power-in-g (a (rot n)) (n 1) (g (sym n)))
				(:instance power-in-g (a (rot n)) (n i) (g (sym n)))
				(:instance power-in-g (a (rot n)) (n (mod (+ -1 (- i) j n) n)) (g (sym n)))
                                (:instance mod-power (i (+ (- i) j n)))
				(:instance power-comp-perm-assoc (x (ref n)) (y (rot n))
					   (i (mod (+ -1 (- i) j n) n)))))
	  ("Subgoal *1/1" :in-theory (enable e)
			  :use ((:instance comp-perm-simplify-right
					   (i (mod (+ n (- i) j) n)))
				(:instance power-mod-minus)
				(:instance group-right-identity (x (ref n)) (g (sym n)))))))

(defthmd power-simplify
    (implies (and (posp n) (natp i) (natp j)
		  (member-equal x (slist n))
		  (member-equal y (slist n)))
	     (equal (comp-perm (comp-perm x (power y i (sym n)) n)
			       (power y j (sym n)) n)
		    (comp-perm x (power y (+ i j) (sym n)) n)))
  :hints (("Goal" :use ((:instance sym-assoc (y (power y i (sym n)))
				   (z (power y j (sym n))))
			(:instance power+ (a y) (n i) (m j) (g (sym n)))
			(:instance power-in-g (a y) (n i) (g (sym n)))
			(:instance power-in-g (a y) (n j) (g (sym n)))))))

(defthmd comp-perm-2-simplify
    (implies (and (natp n) (>= n 3) (natp i) (natp j) (< i n) (< j n))
	     (equal (comp-perm (comp-perm (ref n) (power (rot n) i (sym n)) n)
			       (comp-perm (ref n) (power (rot n) j (sym n)) n) n)
		    (power (rot n) (mod (+ n (- i) j) n) (sym n))))
  :hints (("Goal" :in-theory (disable sym-assoc)
		  :use ((:instance sym-assoc (x (ref n)) (y (power (rot n) i (sym n)))
				   (z (comp-perm (ref n) (power (rot n) j (sym n)) n)))
			(:instance sym-assoc (x (ref n)) (y (ref n))
				   (z (power (rot n) (mod (+ (- i) j n) n) (sym n))))
			comp-perm-simplify
			inv-perm-ref
			(:instance sym-inverse (x (ref n)))
			(:instance power-in-g (a (rot n)) (n (mod (+ (- i) j n) n)) (g (sym n)))
			(:instance power-in-g (a (rot n)) (n i) (g (sym n)))
			(:instance power-in-g (a (rot n)) (n j) (g (sym n)))))))


; we need to establish if an element is in certain forms, they are a member of delts
(defthmd member-power-delts
    (implies (and (posp n) (natp i))
	     (member-equal (power (rot n) i (sym n)) (delts n))))

(defthmd member-list-map-comp-perm
    (implies (and (member-equal y l) (member-equal x (slist n))
		  (sublistp l (slist n)) (posp n))
	     (member-equal (comp-perm x y n)
			   (map-comp-perm x l n))))

(defthmd member-map-comp-perm-converse
    (implies (and (posp n) (natp i))
	     (member-equal (comp-perm (ref n) (power (rot n) i (sym n)) n)
			   (map-comp-perm (ref n) (powers (rot n) (sym n)) n)))
  :hints (("Goal" :use ((:instance member-list-map-comp-perm
				   (x (ref n)) (y (power (rot n) i (sym n)))
				   (l (powers (rot n) (sym n))))
			(:instance sublistp-powers (g (sym n)) (a (rot n)))))))

;; lemma to reduce num of hints needed in the closure proof
(defthmd member-delts-slist
    (implies (and (posp n) (member-equal x (delts n)))
	     (member-equal x (slist n)))
  :hints (("Goal" :in-theory (disable delts)
		  :use ((:instance member-sublist (l (delts n)) (m (slist n)))))))

;;   `(implies (and ,cond (member-equal x ,elts) (member-equal y ,elts))
;;             (member-equal (op x y ,grp) ,elts))
(defthm delts-elts-closed
    (implies (and (natp n) (>= n 3)
		  (member-equal x (delts n))
		  (member-equal y (delts n)))
	     (member-equal (comp-perm x y n) (delts n)))
  :hints (("Goal" :in-theory (disable sym-op-rewrite)
		  :use (sym-op-rewrite
			member-delts-slist
			(:instance member-delts-slist (x y))))
	  ("Subgoal 3" :use ((:instance member-map-comp-perm (p y))
			     (:instance member-powers-rot (p x))
			     (:instance comp-perm-simplify
					(i (index x (powers (rot n) (sym n))))
					(j (index y (map-comp-perm (ref n) (powers (rot n) (sym n)) n))))
			     (:instance member-map-comp-perm-converse
					(i (mod (+ n (- (index x (powers (rot n) (sym n))))
						   (index y (map-comp-perm (ref n) (powers (rot n) (sym n)) n)))
						n)))))
	  ("Subgoal 2" :use ((:instance member-map-comp-perm (p x))
			     (:instance member-powers-rot (p y))
			     (:instance member-sublist (x y) (l (delts n)) (m (slist n)))
			     (:instance member-sublist (l (delts n)) (m (slist n)))
			     (:instance power-simplify (x (ref n)) (y (rot n))
					(i (index x (map-comp-perm (ref n) (powers (rot n) (sym n)) n)))
					(j (index y (powers (rot n) (sym n)))))
			     (:instance member-map-comp-perm-converse
					(i (+ (index x (map-comp-perm (ref n) (powers (rot n) (sym n)) n))
					      (index y (powers (rot n) (sym n))))))))
	  ("Subgoal 1" :use ((:instance member-map-comp-perm (p x))
			     (:instance member-map-comp-perm (p y))
			     (:instance comp-perm-2-simplify
					(i (index x (map-comp-perm (ref n) (powers (rot n) (sym n)) n)))
					(j (index y (map-comp-perm (ref n) (powers (rot n) (sym n)) n))))
			     (:instance power-member (a (rot n)) (g (sym n))
					(n (mod (+ n (- (index x (map-comp-perm (ref n) (powers (rot n) (sym n)) n)))
						   (index y (map-comp-perm (ref n) (powers (rot n) (sym n)) n)))
						n)))
			     (:instance member-power-delts
					(i (mod (+ n (- (index x (map-comp-perm (ref n) (powers (rot n) (sym n)) n)))
						   (index y (map-comp-perm (ref n) (powers (rot n) (sym n)) n)))
						n)))))))

;;   `(implies (and ,cond (member-equal x ,elts))
;;             (member-equal (inv x ,grp) ,elts))
(defthmd comp-perm-squared-eq-e
    (implies (and (natp n) (>= n 3) (< i n) (natp i))
	     (equal (comp-perm (comp-perm (ref n) (power (rot n) i (sym n)) n)
			       (comp-perm (ref n) (power (rot n) i (sym n)) n) n)
		    (e (sym n))))
  :hints (("Goal" :use ((:instance comp-perm-2-simplify (j i))))))

(defthmd comp-perm-inv-eq
    (implies (and (natp n) (>= n 3) (< i n) (natp i))
	     (equal (inv-perm (comp-perm (ref n) (power (rot n) i (sym n)) n) n)
		    (comp-perm (ref n) (power (rot n) i (sym n)) n)))
  :hints (("Goal" :in-theory (enable e)
		  :use ((:instance sym-inverse
				   (x (comp-perm (ref n) (power (rot n) i (sym n)) n)))
			comp-perm-squared-eq-e
			member-map-comp-perm-converse
		        (:instance member-delts-slist
				   (x (comp-perm (ref n) (power (rot n) i (sym n)) n)))
			(:instance right-cancel (g (sym n))
				   (a (comp-perm (ref n) (power (rot n) i (sym n)) n))
				   (x (inv-perm (comp-perm (ref n) (power (rot n) i (sym n)) n) n))
				   (y (comp-perm (ref n) (power (rot n) i (sym n)) n)))))))

(defthm delts-elts-inverse
      (implies (and (natp n) (>= n 3)
                    (member-equal x (delts n)))
               (member-equal (inv-perm x n) (delts n)))
  :hints (("Subgoal 2" :in-theory (disable inv-power sym-inv-rewrite)
		       :use ((:instance inv-power (a (rot n)) (g (sym n)))
			     sym-inv-rewrite
			     member-delts-slist))
	  ("Subgoal 1" :use ((:instance member-map-comp-perm (p x))
			(:instance member-map-comp-perm-converse
		 		   (i (index x (map-comp-perm (ref n) (powers (rot n) (sym n)) n))))
			(:instance comp-perm-inv-eq 
				   (i (index x (map-comp-perm (ref n) (powers (rot n) (sym n)) n))))))))

			
(in-theory (disable delts))
(in-theory (enable e))
(defsubgroup dihedral (n)
  (sym n)
  (and (natp n) (>= n 3))
  (delts n))
(in-theory (enable delts comp-perm-ref-rot))
(in-theory (disable e))
