/*  RetroArch - A frontend for libretro.
 *  Copyright (C) 2011-2017 - Daniel De Matteis
 *  Copyright (C) 2013-2014 - Jason Fetters
 * 
 *  RetroArch is free software: you can redistribute it and/or modify it under the terms
 *  of the GNU General Public License as published by the Free Software Found-
 *  ation, either version 3 of the License, or (at your option) any later version.
 *
 *  RetroArch is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 *  without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 *  PURPOSE.  See the GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along with RetroArch.
 *  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdint.h>
#include <unistd.h>

#ifdef HAVE_CONFIG_H
#include "../../config.h"
#endif

#include "../input_config.h"
#include "../input_keymaps.h"
#include "../input_keyboard.h"

#include "../../runloop.h"
#include "../../driver.h"

#include "keyboard_event_apple.h"

#if TARGET_OS_IPHONE
static bool small_keyboard_active;

#define HIDKEY(X) X
#else
/* Taken from https://github.com/depp/keycode,
 * check keycode.h for license. */

static const unsigned char MAC_NATIVE_TO_HID[128] = {
   4, 22,  7,  9, 11, 10, 29, 27,  6, 25,255,  5, 20, 26,  8, 21,
   28, 23, 30, 31, 32, 33, 35, 34, 46, 38, 36, 45, 37, 39, 48, 18,
   24, 47, 12, 19, 40, 15, 13, 52, 14, 51, 49, 54, 56, 17, 16, 55,
   43, 44, 53, 42,255, 41,231,227,225, 57,226,224,229,230,228,255,
   108, 99,255, 85,255, 87,255, 83,255,255,255, 84, 88,255, 86,109,
   110,103, 98, 89, 90, 91, 92, 93, 94, 95,111, 96, 97,255,255,255,
   62, 63, 64, 60, 65, 66,255, 68,255,104,107,105,255, 67,255, 69,
   255,106,117, 74, 75, 76, 61, 77, 59, 78, 58, 80, 79, 81, 82,255
};

#define HIDKEY(X) (X < 128) ? MAC_NATIVE_TO_HID[X] : 0
#endif

static uint32_t apple_key_state[MAX_KEYS];

#if TARGET_OS_IPHONE
static bool handle_small_keyboard(unsigned* code, bool down)
{
   static uint8_t mapping[128];
   static bool map_initialized;
   static const struct { uint8_t orig; uint8_t mod; } mapping_def[] =
   {
      { KEY_Grave,      KEY_Escape     }, { KEY_1,          KEY_F1         },
      { KEY_2,          KEY_F2         }, { KEY_3,          KEY_F3         },
      { KEY_4,          KEY_F4         }, { KEY_5,          KEY_F5         },
      { KEY_6,          KEY_F6         }, { KEY_7,          KEY_F7         },
      { KEY_8,          KEY_F8         }, { KEY_9,          KEY_F9         },
      { KEY_0,          KEY_F10        }, { KEY_Minus,      KEY_F11        },
      { KEY_Equals,     KEY_F12        }, { KEY_Up,         KEY_PageUp     },
      { KEY_Down,       KEY_PageDown   }, { KEY_Left,       KEY_Home       },
      { KEY_Right,      KEY_End        }, { KEY_Q,          KP_7           },
      { KEY_W,          KP_8           }, { KEY_E,          KP_9           },
      { KEY_A,          KP_4           }, { KEY_S,          KP_5           },
      { KEY_D,          KP_6           }, { KEY_Z,          KP_1           },
      { KEY_X,          KP_2           }, { KEY_C,          KP_3           },
      { 0 }
   };
   unsigned translated_code  = 0;

   if (!map_initialized)
   {
      int i;
      for (i = 0; mapping_def[i].orig; i ++)
         mapping[mapping_def[i].orig] = mapping_def[i].mod;
      map_initialized = true;
   }

   if (*code == KEY_RightShift)
   {
      small_keyboard_active = down;
      *code = 0;
      return true;
   }

   translated_code = (*code < 128) ? mapping[*code] : 0;

   /* Allow old keys to be released. */
   if (!down && apple_key_state[*code])
      return false;

   if ((!down && apple_key_state[translated_code]) ||
         small_keyboard_active)
   {
      *code = translated_code;
      return true;
   }

   return false;
}

typedef struct icade_map
{
   bool up;
   enum retro_key key;
} icade_map_t;

#define MAX_ICADE_PROFILES 4
#define MAX_ICADE_KEYS     0x100

static icade_map_t icade_maps[MAX_ICADE_PROFILES][MAX_ICADE_KEYS];

static bool handle_icade_event(unsigned *code, bool *keydown)
{
   settings_t *settings = config_get_ptr();
   static bool initialized = false;
   bool ret = false;
   unsigned kb_type_idx = settings->input.keyboard_gamepad_mapping_type;

   if (!initialized)
   {
      unsigned i;
      unsigned j = 0;

      for (j = 0; j < MAX_ICADE_PROFILES; j++)
      {
         for (i = 0; i < MAX_ICADE_KEYS; i++)
         {
            icade_maps[j][i].key = RETROK_UNKNOWN;
            icade_maps[j][i].up  = false;
         }
      }

      /* iPega PG-9017 */
      j = 1;

      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_a)].key = RETROK_LEFT;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_q)].key = RETROK_LEFT;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_c)].key = RETROK_RIGHT;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_d)].key = RETROK_RIGHT;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_e)].key = RETROK_UP;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_w)].key = RETROK_UP;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_x)].key = RETROK_DOWN;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_z)].key = RETROK_DOWN;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_f)].key = RETROK_z;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_u)].key = RETROK_z;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_i)].key = RETROK_q;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_m)].key = RETROK_q;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_j)].key = RETROK_a;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_n)].key = RETROK_a;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_k)].key = RETROK_w;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_p)].key = RETROK_w;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_h)].key = RETROK_x;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_r)].key = RETROK_x;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_y)].key = RETROK_s;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_t)].key = RETROK_s;

      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_e)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_z)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_q)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_c)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_f)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_m)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_t)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_n)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_p)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_r)].up  = true;

      /* 8-bitty */
      j = 2;

      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_a)].key = RETROK_LEFT;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_q)].key = RETROK_LEFT;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_c)].key = RETROK_RIGHT;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_d)].key = RETROK_RIGHT;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_e)].key = RETROK_UP;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_w)].key = RETROK_UP;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_x)].key = RETROK_DOWN;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_z)].key = RETROK_DOWN;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_h)].key = RETROK_q;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_r)].key = RETROK_q;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_j)].key = RETROK_w;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_n)].key = RETROK_w;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_i)].key = RETROK_a;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_m)].key = RETROK_a;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_k)].key = RETROK_z;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_p)].key = RETROK_z;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_y)].key = RETROK_RSHIFT;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_t)].key = RETROK_RSHIFT;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_u)].key = RETROK_RETURN;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_f)].key = RETROK_RETURN;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_l)].key = RETROK_x;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_v)].key = RETROK_x;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_o)].key = RETROK_s;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_g)].key = RETROK_s;

      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_e)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_z)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_q)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_c)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_r)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_n)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_m)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_p)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_t)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_f)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_v)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_g)].up  = true;

      /* SNES30 8bitDo */
      j = 3;

      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_e)].key = RETROK_UP;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_w)].key = RETROK_UP;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_x)].key = RETROK_DOWN;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_z)].key = RETROK_DOWN;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_a)].key = RETROK_LEFT;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_q)].key = RETROK_LEFT;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_c)].key = RETROK_RIGHT;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_d)].key = RETROK_RIGHT;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_u)].key = RETROK_x;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_f)].key = RETROK_x;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_h)].key = RETROK_z;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_r)].key = RETROK_z;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_y)].key = RETROK_a;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_t)].key = RETROK_a;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_j)].key = RETROK_s;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_n)].key = RETROK_s;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_k)].key = RETROK_q;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_p)].key = RETROK_q;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_i)].key = RETROK_w;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_m)].key = RETROK_w;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_l)].key = RETROK_RSHIFT;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_v)].key = RETROK_RSHIFT;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_o)].key = RETROK_RETURN;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_g)].key = RETROK_RETURN;

      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_v)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_g)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_e)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_z)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_q)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_c)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_r)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_f)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_n)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_t)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_p)].up  = true;
      icade_maps[j][input_keymaps_translate_rk_to_keysym(RETROK_m)].up  = true;

      initialized = true;
   }

   if ((*code < 0x20) && (icade_maps[kb_type_idx][*code].key != RETROK_UNKNOWN))
   {
      *keydown     = icade_maps[kb_type_idx][*code].up ? false : true;
      ret          = true;
      *code        = input_keymaps_translate_rk_to_keysym(icade_maps[kb_type_idx][*code].key);
   }

   return ret;
}
#endif

void apple_input_keyboard_event(bool down,
      unsigned code, uint32_t character, uint32_t mod, unsigned device)
{
#if TARGET_OS_IPHONE
   settings_t *settings = config_get_ptr();
#endif

   code = HIDKEY(code);

#if TARGET_OS_IPHONE
   if (settings->input.keyboard_gamepad_enable)
   {
      if (handle_icade_event(&code, &down))
         character = 0;
      else
         code      = 0;
   }
   else if (settings->input.small_keyboard_enable)
   {
      if (handle_small_keyboard(&code, down))
         character = 0;
   }
#endif

   if (code == 0 || code >= MAX_KEYS)
      return;

   apple_key_state[code] = down;

   input_keyboard_event(down,
         input_keymaps_translate_keysym_to_rk(code),
         character, (enum retro_mod)mod, device);
}

int16_t apple_input_is_pressed(unsigned port_num,
   const struct retro_keybind *binds, unsigned id)
{
   if (id < RARCH_BIND_LIST_END)
   {
      const struct retro_keybind *bind = &binds[id];
      unsigned bit = input_keymaps_translate_rk_to_keysym(bind->key);
      return apple_key_state[bit];
   }
   return 0;
}

int16_t apple_keyboard_state(unsigned id)
{
   unsigned bit = input_keymaps_translate_rk_to_keysym((enum retro_key)id);
   return (id < RETROK_LAST) && apple_key_state[bit];
}

int32_t apple_keyboard_find_any_key(void)
{
   unsigned i;

   for (i = 0; apple_key_name_map[i].hid_id; i++)
      if (apple_key_state[apple_key_name_map[i].hid_id])
         return apple_key_name_map[i].hid_id;

   return 0;
}

void apple_keyboard_free(void)
{
   unsigned i;
   for (i = 0; i < MAX_KEYS; i++)
      apple_key_state[i] = 0;
}
