/*	memory.c
	Copyright (C) 2004-2007 Mark Tyler and Dmitry Groshev

	This file is part of rgbPaint.

	rgbPaint is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	rgbPaint is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with rgbPaint in the file COPYING.
*/

#include <math.h>
#include <stdlib.h>
#include <string.h>

#include "global.h"

#include "memory.h"
#include "png.h"
#include "mainwindow.h"
#include "otherwindow.h"
#include "mygtk.h"
#include "canvas.h"
#include "toolbar.h"
#include "viewer.h"


//char *channames[NUM_CHANNELS + 1], *allchannames[NUM_CHANNELS + 1];

/// IMAGE

char mem_filename[256];			// File name of file loaded/saved
chanlist mem_img;			// Array of pointers to image channels
int mem_channel = CHN_IMAGE;		// Current active channel
int mem_img_bpp;			// Bytes per pixel = 1 or 3
int mem_changed;			// Changed since last load/save flag 0=no, 1=changed
int mem_width, mem_height;
float mem_icx = 0.5, mem_icy = 0.5;	// Current centre x,y
int mem_ics;				// Has the centre been set by the user? 0=no 1=yes
int mem_background = 180;		// Non paintable area

unsigned char *mem_clipboard;		// Pointer to clipboard data
unsigned char *mem_clip_mask;		// Pointer to clipboard mask
unsigned char *mem_clip_alpha;		// Pointer to clipboard alpha
int mem_clip_bpp;			// Bytes per pixel
int mem_clip_w = -1, mem_clip_h = -1;	// Clipboard geometry
int mem_clip_x = -1, mem_clip_y = -1;	// Clipboard location on canvas
int mem_nudge = 10;			// Nudge pixels per SHIFT+Arrow key during selection/paste

int mem_preview;			// Preview an RGB change
int mem_prev_bcsp[6];			// BR, CO, SA, POSTERIZE, Hue

undo_item mem_undo_im[MAX_UNDO];	// Pointers to undo images + current image being edited

int mem_undo_pointer;		// Pointer to currently used image on canas/screen
int mem_undo_done;		// Undo images that we have behind current image (i.e. possible UNDO)
int mem_undo_redo;		// Undo images that we have ahead of current image (i.e. possible REDO)
int mem_undo_limit = 32;	// Max MB memory allocation limit

/// GRID

int mem_show_grid = TRUE, mem_grid_min = 8;	// Boolean show toggle & minimum zoom to show it at
unsigned char mem_grid_rgb[3] = {50,50,50};	// RGB colour of grid

/// PREVIEW/TOOLS

int tool_type = TOOL_SQUARE;		// Currently selected tool
int brush_type;				// The current brush - same as tool_type if square/circle/spray
int tool_size = 1, tool_flow = 1;
int pen_down;				// Are we drawing? - Used to see if we need to do an UNDO
int tool_ox, tool_oy;			// Previous tool coords - used by continuous mode
int mem_continuous = TRUE;		// Area we painting the static shapes continuously?



/// FILE

int mem_jpeg_quality = 85;		// JPEG quality setting

/// PALETTE

png_color mem_pal[256];			// RGB entries for all 256 palette colours
int mem_cols;				// Number of colours in the palette: 2..256 or 0 for no image
int mem_col_[2] = { 1, 0 };		// Index for colour A & B
png_color mem_col_24[2];		// RGB for colour A & B

int mem_pal_def_i = 256;		// Items in default palette

png_color mem_pal_def[256]={		// Default palette entries for new image
/// All RGB in 3 bits per channel. i.e. 0..7 - multiply by 255/7 for full RGB ..
/// .. or: int lookup[8] = {0, 36, 73, 109, 146, 182, 219, 255};

/// Primary colours = 8

{0,0,0}, {2,2,2}, {3,3,3}, {4,4,4}, {5,5,5}, {7,7,7}, {3,5,7}, {1,3,7}, {0,0,7}, {0,4,4},
{7,0,7}, {7,0,0}, {4,0,0}, {0,3,0}, {0,5,0}, {0,7,0}, {7,7,0}, {7,5,0}, {5,4,2}, {7,5,5}, 




{0,6,0}, {0,5,0}, {0,4,0}, {0,3,0}, {0,2,0}, {0,1,0},
{6,6,0}, {5,5,0}, {4,4,0}, {3,3,0}, {2,2,0}, {1,1,0},
{0,0,6}, {0,0,5}, {0,0,4}, {0,0,3}, {0,0,2}, {0,0,1},
{6,0,6}, {5,0,5}, {4,0,4}, {3,0,3}, {2,0,2}, {1,0,1},
{0,6,6}, {0,5,5}, {0,4,4}, {0,3,3}, {0,2,2}, {0,1,1},

/// Shading triangles: 6 x 21 = 126
/// RED
{7,6,6}, {6,5,5}, {5,4,4}, {4,3,3}, {3,2,2}, {2,1,1},
{7,5,5}, {6,4,4}, {5,3,3}, {4,2,2}, {3,1,1},
{7,4,4}, {6,3,3}, {5,2,2}, {4,1,1},
{7,3,3}, {6,2,2}, {5,1,1},
{7,2,2}, {6,1,1},
{7,1,1},

/// GREEN
{6,7,6}, {5,6,5}, {4,5,4}, {3,4,3}, {2,3,2}, {1,2,1},
{5,7,5}, {4,6,4}, {3,5,3}, {2,4,2}, {1,3,1},
{4,7,4}, {3,6,3}, {2,5,2}, {1,4,1},
{3,7,3}, {2,6,2}, {1,5,1},
{2,7,2}, {1,6,1},
{1,7,1},

/// BLUE
{6,6,7}, {5,5,6}, {4,4,5}, {3,3,4}, {2,2,3}, {1,1,2},
{5,5,7}, {4,4,6}, {3,3,5}, {2,2,4}, {1,1,3},
{4,4,7}, {3,3,6}, {2,2,5}, {1,1,4},
{3,3,7}, {2,2,6}, {1,1,5},
{2,2,7}, {1,1,6},
{1,1,7},

/// YELLOW (red + green)
{7,7,6}, {6,6,5}, {5,5,4}, {4,4,3}, {3,3,2}, {2,2,1},
{7,7,5}, {6,6,4}, {5,5,3}, {4,4,2}, {3,3,1},
{7,7,4}, {6,6,3}, {5,5,2}, {4,4,1},
{7,7,3}, {6,6,2}, {5,5,1},
{7,7,2}, {6,6,1},
{7,7,1},

/// MAGENTA (red + blue)
{7,6,7}, {6,5,6}, {5,4,5}, {4,3,4}, {3,2,3}, {2,1,2},
{7,5,7}, {6,4,6}, {5,3,5}, {4,2,4}, {3,1,3},
{7,4,7}, {6,3,6}, {5,2,5}, {4,1,4},
{7,3,7}, {6,2,6}, {5,1,5},
{7,2,7}, {6,1,6},
{7,1,7},

/// CYAN (blue + green)
{6,7,7}, {5,6,6}, {4,5,5}, {3,4,4}, {2,3,3}, {1,2,2},
{5,7,7}, {4,6,6}, {3,5,5}, {2,4,4}, {1,3,3},
{4,7,7}, {3,6,6}, {2,5,5}, {1,4,4},
{3,7,7}, {2,6,6}, {1,5,5},
{2,7,7}, {1,6,6},
{1,7,7},


/// Scales: 11 x 6 = 66

/// RGB
{7,6,5}, {6,5,4}, {5,4,3}, {4,3,2}, {3,2,1}, {2,1,0},
{7,5,4}, {6,4,3}, {5,3,2}, {4,2,1}, {3,1,0},

/// RBG
{7,5,6}, {6,4,5}, {5,3,4}, {4,2,3}, {3,1,2}, {2,0,1},
{7,4,5}, {6,3,4}, {5,2,3}, {4,1,2}, {3,0,1},

/// BRG
{6,5,7}, {5,4,6}, {4,3,5}, {3,2,4}, {2,1,3}, {1,0,2},
{5,4,7}, {4,3,6}, {3,2,5}, {2,1,4}, {1,0,3},

/// BGR
{5,6,7}, {4,5,6}, {3,4,5}, {2,3,4}, {1,2,3}, {0,1,2},
{4,5,7}, {3,4,6}, {2,3,5}, {1,2,4}, {0,1,3},

/// GBR
{5,7,6}, {4,6,5}, {3,5,4}, {2,4,3}, {1,3,2}, {0,2,1},
{4,7,5}, {3,6,4}, {2,5,3}, {1,4,2}, {0,3,1},

/// GRB
{6,7,5}, {5,6,4}, {4,5,3}, {3,4,2}, {2,3,1}, {1,2,0},
{5,7,4}, {4,6,3}, {3,5,2}, {2,4,1}, {1,3,0},

/// Misc
{7,5,0}, {6,4,0}, {5,3,0}, {4,2,0},		// Oranges
{7,0,5}, {6,0,4}, {5,0,3}, {4,0,2},		// Red Pink
{0,5,7}, {0,4,6}, {0,3,5}, {0,2,4},		// Blues
{0,0,0}, {0,0,0}

/// End: Primary (8) + Fades (42) + Shades (126) + Scales (66) + Misc (14) = 256
};

/* Set initial state of image variables */
void init_istate()
{
	notify_unchanged();

	mem_col_A = 0;
	mem_col_B = 0;
	mem_col_A24 = mem_pal[mem_col_A];
	mem_col_B24 = mem_pal[mem_col_B];
}

void undo_free_x(undo_item *undo)
{
	int i;

	for (i = 0; i < NUM_CHANNELS; i++)
	{
		if (!undo->img[i]) continue;
		if (undo->img[i] != (void *)(-1)) free(undo->img[i]);
		undo->img[i] = NULL;
	}
}

void undo_free(int idx)
{
	undo_free_x(&mem_undo_im_[idx]);
}

void mem_clear()
{
	int i;

	for (i = 0; i < MAX_UNDO; i++)		// Release old UNDO images
		undo_free(i);
	memset(mem_img, 0, sizeof(chanlist));	// Already freed along with UNDO
	mem_undo_pointer = mem_undo_done = mem_undo_redo = 0;
}

/* Allocate space for new image, removing old if needed */
int mem_new( int width, int height, int bpp, int cmask )
{
	unsigned char *res;
	undo_item *undo = &mem_undo_im_[0];
	int i, j = width * height;

	mem_clear();

	res = mem_img[CHN_IMAGE] = malloc(j * bpp);
	for (i = CHN_ALPHA; res && (cmask > CMASK_FOR(i)); i++)
	{
		if (!(cmask & CMASK_FOR(i))) continue;
		res = mem_img[i] = malloc(j);
	}
	if (!res)	// Not enough memory
	{
		for (i = 0; i < NUM_CHANNELS; i++)
			free(mem_img[i]);
		memset(mem_img, 0, sizeof(chanlist));
		width = height = 8; // 8x8 is bound to work!
		j = width * height;
		mem_img[CHN_IMAGE] = malloc(j * bpp);
	}

	i = 255;
	memset(mem_img[CHN_IMAGE], i, j * bpp);

	mem_width = width;
	mem_height = height;
	mem_img_bpp = bpp;
	mem_channel = CHN_IMAGE;

	memcpy(undo->img, mem_img, sizeof(chanlist));
	undo->cols = mem_cols;
	undo->bpp = mem_img_bpp;
	undo->width = width;
	undo->height = height;
	mem_pal_copy(undo->pal, mem_pal);

//	mem_xpm_trans = mem_xbm_hot_x = mem_xbm_hot_y = -1;

	return (!res);
}

/* Get address of previous channel data (or current if none) */
unsigned char *mem_undo_previous(int channel)
{
	unsigned char *res;
	int i;

	i = mem_undo_pointer ? mem_undo_pointer - 1 : MAX_UNDO - 1;
	res = mem_undo_im_[i].img[channel];
	if (!res || (res == (void *)(-1)))
		res = mem_img[channel];	// No undo so use current
	return (res);
}

void lose_oldest()				// Lose the oldest undo image
{						// Pre-requisite: mem_undo_done > 0
	undo_free((mem_undo_pointer - mem_undo_done + MAX_UNDO) % MAX_UNDO);
	mem_undo_done--;
}

int undo_next_core(int mode, int new_width, int new_height, int new_bpp, int cmask)
{
	undo_item *undo;
	unsigned char *img;
	chanlist holder;
	int i, j, k, mem_req, mem_lim;

	notify_changed();
	if (pen_down && (mode & UC_PENDOWN)) return (0);
	pen_down = mode & UC_PENDOWN ? 1 : 0;
//printf("Old undo # = %i\n", mem_undo_pointer);

	/* Release redo data */
	if (mem_undo_redo)
	{
		k = mem_undo_pointer;
		for (i = 0; i < mem_undo_redo; i++)
		{
			k = (k + 1) % MAX_UNDO;
			undo_free(k);
		}
		mem_undo_redo = 0;
	}

	mem_req = 0;
	if (cmask && !(mode & UC_DELETE))
	{
		for (i = j = 0; i < NUM_CHANNELS; i++)
		{
			if ((mem_img[i] || (mode & UC_CREATE))
				&& (cmask & (1 << i))) j++;
		}
		if (cmask & CMASK_IMAGE) j += new_bpp - 1;
		mem_req = (new_width * new_height + 32) * j;
	}
//	mem_lim = (mem_undo_limit * (1024 * 1024)) / (layers_total + 1);
	mem_lim = (mem_undo_limit * (1024 * 1024));

	/* Mem limit exceeded - drop oldest */
	while (mem_used() + mem_req > mem_lim)
	{
		if (!mem_undo_done)
		{
			/* Fail if not enough memory */
			if (!(mode & UC_DELETE)) return (1);
		}
		else lose_oldest();
	}

	/* Fill undo frame */
	undo = &mem_undo_im_[mem_undo_pointer];
	for (i = 0; i < NUM_CHANNELS; i++)
	{
		img = mem_img[i];
		if (img && !(cmask & (1 << i))) img = (void *)(-1);
		undo->img[i] = img;
	}
	undo->cols = mem_cols;
	mem_pal_copy(undo->pal, mem_pal);
	undo->width = mem_width;
	undo->height = mem_height;
	undo->bpp = mem_img_bpp;

	/* Duplicate affected channels */
	mem_req = new_width * new_height;
	for (i = 0; i < NUM_CHANNELS; i++)
	{
		holder[i] = img = mem_img[i];
		if (!(cmask & (1 << i))) continue;
		if (mode & UC_DELETE)
		{
			holder[i] = NULL;
			continue;
		}
		if (!img && !(mode & UC_CREATE)) continue;
		mem_lim = mem_req;
		if (i == CHN_IMAGE) mem_lim *= new_bpp;
		while (!((img = malloc(mem_lim))))
		{
			if (!mem_undo_done)
			{
				/* Release memory and fail */
				for (j = 0; j < i; j++)
				{
					if (holder[j] != mem_img[j])
						free(holder[j]);
				}
				return (2);
			}
			lose_oldest();
		}
		holder[i] = img;
		/* Copy */
		if (!undo->img[i] || (mode & UC_NOCOPY)) continue;
		memcpy(img, undo->img[i], mem_lim);
	}

	/* Next undo step */
	if (mem_undo_done >= MAX_UNDO - 1)
		undo_free((mem_undo_pointer + 1) % MAX_UNDO);
	else mem_undo_done++;
	mem_undo_pointer = (mem_undo_pointer + 1) % MAX_UNDO;	// New pointer

	/* Commit */
	memcpy(mem_img, holder, sizeof(chanlist));
	mem_width = new_width;
	mem_height = new_height;
	mem_img_bpp = new_bpp;

	undo = &mem_undo_im_[mem_undo_pointer];
	memcpy(undo->img, holder, sizeof(chanlist));
	undo->cols = mem_cols;
	mem_pal_copy(undo->pal, mem_pal);
	undo->width = mem_width;
	undo->height = mem_height;
	undo->bpp = mem_img_bpp;
//printf("New undo # = %i\n\n", mem_undo_pointer);

	return (0);
}

// Call this after a draw event but before any changes to image
int mem_undo_next(int mode)
{
	int cmask = CMASK_ALL, wmode = 0;

	switch (mode)
	{
	case UNDO_TOOL: /* Continuous drawing */
		wmode = UC_PENDOWN;
	case UNDO_DRAW: /* Changes to current channel / RGBA */
		cmask = (mem_channel == CHN_IMAGE) /*&& RGBA_mode*/ ?
			CMASK_RGBA : CMASK_CURR;
		break;
	case UNDO_PASTE: /* Paste to current channel / RGBA */
		wmode = UC_PENDOWN;	/* !!! Workaround for move-with-RMB-pressed */
		cmask = (mem_channel == CHN_IMAGE) && /*!channel_dis[CHN_ALPHA] &&*/
			(mem_clip_alpha /*|| RGBA_mode*/) ? CMASK_RGBA : CMASK_CURR;
		break;
	}
	return (undo_next_core(wmode, mem_width, mem_height, mem_img_bpp, cmask));
}

void mem_undo_swap(int old, int new)
{
	undo_item *curr, *prev;
	int i;

	curr = &mem_undo_im_[old];
	prev = &mem_undo_im_[new];

	for (i = 0; i < NUM_CHANNELS; i++)
	{
		curr->img[i] = mem_img[i];
		if (prev->img[i] == (void *)(-1)) curr->img[i] = (void *)(-1);
		else mem_img[i] = prev->img[i];
	}

	curr->width = mem_width;
	curr->height = mem_height;
	curr->cols = mem_cols;
	curr->bpp = mem_img_bpp;
	mem_pal_copy(curr->pal, mem_pal);

	mem_width = prev->width;
	mem_height = prev->height;
	mem_cols = prev->cols;
	mem_img_bpp = prev->bpp;
	mem_pal_copy(mem_pal, prev->pal);

	if ( mem_col_A >= mem_cols ) mem_col_A = 0;
	if ( mem_col_B >= mem_cols ) mem_col_B = 0;
	if (!mem_img[mem_channel]) mem_channel = CHN_IMAGE;
}

void mem_undo_backward()		// UNDO requested by user
{
	int i;

	if ( mem_undo_done > 0 )
	{
//printf("UNDO!!! Old undo # = %i\n", mem_undo_pointer);
		i = mem_undo_pointer;
		mem_undo_pointer = (mem_undo_pointer - 1 + MAX_UNDO) % MAX_UNDO;	// New pointer
		mem_undo_swap(i, mem_undo_pointer);

		mem_undo_done--;
		mem_undo_redo++;
//printf("New undo # = %i\n\n", mem_undo_pointer);
	}
	pen_down = 0;
}

void mem_undo_forward()			// REDO requested by user
{
	int i;

	if ( mem_undo_redo > 0 )
	{
//printf("REDO!!! Old undo # = %i\n", mem_undo_pointer);
		i = mem_undo_pointer;
		mem_undo_pointer = (mem_undo_pointer + 1) % MAX_UNDO;		// New pointer
		mem_undo_swap(i, mem_undo_pointer);

		mem_undo_done++;
		mem_undo_redo--;
//printf("New undo # = %i\n\n", mem_undo_pointer);
	}
	pen_down = 0;
}

int mem_undo_size(undo_item *undo)
{
	int i, j, k, total = 0;

	for (i = 0; i < MAX_UNDO; i++)
	{
		k = undo->width * undo->height + 32;
		for (j = 0; j < NUM_CHANNELS; j++)
		{
			if (!undo->img[j] || (undo->img[j] == (void *)(-1)))
				continue;
			total += j == CHN_IMAGE ? k * undo->bpp : k;
		}
		undo++;
	}

	return total;
}

int valid_file( char *filename )		// Can this file be opened for reading?
{
	FILE *fp;

	fp = fopen(filename, "r");
	if ( fp == NULL ) return -1;
	else
	{
		fclose( fp );
		return 0;
	}
}


char *grab_memory( int size, char byte )	// Malloc memory, reset all bytes
{
	char *chunk;

	chunk = malloc( size );
	
	if (chunk) memset(chunk, byte, size);

	return chunk;
}

void mem_init()					// Initialise memory
{
	int i, lookup[8] = {0, 36, 73, 109, 146, 182, 219, 255};


	for ( i=0; i<256; i++ )		// Load up normal palette defaults
	{
		mem_pal_def[i].red = lookup[mem_pal_def[i].red];
		mem_pal_def[i].green = lookup[mem_pal_def[i].green];
		mem_pal_def[i].blue = lookup[mem_pal_def[i].blue];
	}

	for ( i=1; i<5; i++ )		// Create smooth greyscale gradient
	{
		mem_pal_def[i].red = (i*255)/5;
		mem_pal_def[i].green = (i*255)/5;
		mem_pal_def[i].blue = (i*255)/5;
	}

	initialize_text();
	mem_pal_copy( mem_pal, mem_pal_def );
	mem_cols = mem_pal_def_i;
}


static unsigned char gamma_table[256], bc_table[256];
static int last_gamma, last_br, last_co;

void do_transform(int start, int step, int cnt, unsigned char *mask,
	unsigned char *imgr, unsigned char *img0)
{
	static int ixx[7] = {0, 1, 2, 0, 1, 2, 0};
	static int posm[9] = {0, 0xFF00, 0x5500, 0x2480, 0x1100,
				 0x0840, 0x0410, 0x0204, 0};
	int do_gamma, do_bc, do_sa;
	double w;
	unsigned char rgb[3];
	int br, co, sa, ps, pmul;
	int dH, sH, tH, ix0, ix1, ix2, c0, c1, c2, dc = 0;
	int i, j, r, g, b, ofs3, opacity, op0 = 0, op1 = 0, op2 = 0, ops;

	cnt = start + step * cnt;
	ops = op0 + op1 + op2;

	br = mem_prev_bcsp[0] * 255;
	co = mem_prev_bcsp[1];
	if (co > 0) co *= 3;
	co += 100;
	co = (255 * co) / 100;
	sa = (255 * mem_prev_bcsp[2]) / 100;
	dH = sH = mem_prev_bcsp[5];
	ps = 8 - mem_prev_bcsp[3];
	pmul = posm[mem_prev_bcsp[3]];

	do_gamma = mem_prev_bcsp[4] - 100;
	do_bc = br | (co - 255);
	do_sa = sa - 255;

	/* Prepare gamma table */
	if (do_gamma && (do_gamma != last_gamma))
	{
		last_gamma = do_gamma;
		w = 100.0 / (double)mem_prev_bcsp[4];
		for (i = 0; i < 256; i++)
		{
			gamma_table[i] = rint(255.0 * pow((double)i / 255.0, w));
		}
	}
	/* Prepare brightness-contrast table */
	if (do_bc && ((br != last_br) || (co != last_co)))
	{
		last_br = br; last_co = co;
		for (i = 0; i < 256; i++)
		{
			j = ((i + i - 255) * co + (255 * 255)) / 2 + br;
			j = j < 0 ? 0 : j > (255 * 255) ? (255 * 255) : j;
			bc_table[i] = (j + (j >> 8) + 1) >> 8;
		}
	}
	if (dH)
	{
		if (dH < 0) dH += 1530;
		dc = (dH / 510) * 2; dH -= dc * 255;
		if ((sH = dH > 255))
		{
			dH = 510 - dH;
			dc = dc < 4 ? dc + 2 : 0;
		}
	}
	ix0 = ixx[dc]; ix1 = ixx[dc + 1]; ix2 = ixx[dc + 2];

	for (i = start; i < cnt; i += step)
	{
		ofs3 = i * 3;
		rgb[0] = img0[ofs3 + 0];
		rgb[1] = img0[ofs3 + 1];
		rgb[2] = img0[ofs3 + 2];
		opacity = mask[i];
		if (opacity == 255)
		{
			imgr[ofs3 + 0] = rgb[0];
			imgr[ofs3 + 1] = rgb[1];
			imgr[ofs3 + 2] = rgb[2];
			continue;
		}
		/* If we do gamma transform */
		if (do_gamma)
		{
			rgb[0] = gamma_table[rgb[0]];
			rgb[1] = gamma_table[rgb[1]];
			rgb[2] = gamma_table[rgb[2]];
		}
		/* If we do hue transform & colour has a hue */
		if (dH && ((rgb[0] ^ rgb[1]) | (rgb[0] ^ rgb[2])))
		{
			/* Min. component */
			c2 = dc;
			if (rgb[ix2] < rgb[ix0]) c2++;
			if (rgb[ixx[c2]] >= rgb[ixx[c2 + 1]]) c2++;
			/* Actual indices */
			c2 = ixx[c2];
			c0 = ixx[c2 + 1];
			c1 = ixx[c2 + 2];

			/* Max. component & edge dir */
			if ((tH = rgb[c0] <= rgb[c1]))
			{
				c0 = ixx[c2 + 2];
				c1 = ixx[c2 + 1];
			}
			/* Do adjustment */
			j = dH * (rgb[c0] - rgb[c2]) + 127; /* Round up (?) */
			j = (j + (j >> 8) + 1) >> 8;
			r = rgb[c0]; g = rgb[c1]; b = rgb[c2];
			if (tH ^ sH) /* Falling edge */
			{
				rgb[c1] = r = g > j + b ? g - j : b;
				rgb[c2] += j + r - g;
			}
			else /* Rising edge */
			{
				rgb[c1] = b = g < r - j ? g + j : r;
				rgb[c0] -= j + g - b;
			}
		}
		r = rgb[ix0];
		g = rgb[ix1];
		b = rgb[ix2];
		/* If we do brightness/contrast transform */
		if (do_bc)
		{
			r = bc_table[r];
			g = bc_table[g];
			b = bc_table[b];
		}
		/* If we do saturation transform */
		if (sa)
		{
			j = 0.299 * r + 0.587 * g + 0.114 * b;
			r = r * 255 + (r - j) * sa;
			r = r < 0 ? 0 : r > (255 * 255) ? (255 * 255) : r;
			r = (r + (r >> 8) + 1) >> 8;
			g = g * 255 + (g - j) * sa;
			g = g < 0 ? 0 : g > (255 * 255) ? (255 * 255) : g;
			g = (g + (g >> 8) + 1) >> 8;
			b = b * 255 + (b - j) * sa;
			b = b < 0 ? 0 : b > (255 * 255) ? (255 * 255) : b;
			b = (b + (b >> 8) + 1) >> 8;
		}
		/* If we do posterize transform */
		if (ps)
		{
			r = ((r >> ps) * pmul) >> 8;
			g = ((g >> ps) * pmul) >> 8;
			b = ((b >> ps) * pmul) >> 8;
		}
		/* If we do partial masking */
		if (ops || opacity)
		{
			r = r * 255 + (img0[ofs3 + 0] - r) * (opacity | op0);
			r = (r + (r >> 8) + 1) >> 8;
			g = g * 255 + (img0[ofs3 + 1] - g) * (opacity | op1);
			g = (g + (g >> 8) + 1) >> 8;
			b = b * 255 + (img0[ofs3 + 2] - b) * (opacity | op2);
			b = (b + (b >> 8) + 1) >> 8;
		}
		imgr[ofs3 + 0] = r;
		imgr[ofs3 + 1] = g;
		imgr[ofs3 + 2] = b;
	}
}



void set_zoom_centre( int x, int y )
{
	IF_IN_RANGE( x, y )
	{
		mem_icx = ((float) x ) / mem_width;
		mem_icy = ((float) y ) / mem_height;
		mem_ics = 1;
	}
}

int mem_pal_cmp( png_color *pal1, png_color *pal2 )	// Count itentical palette entries
{
	int i, j = 0;

	for ( i=0; i<256; i++ ) if ( pal1[i].red != pal2[i].red ||
				pal1[i].green != pal2[i].green ||
				pal1[i].blue != pal2[i].blue ) j++;

	return j;
}

int mem_used()				// Return the number of bytes used in image + undo
{
	return mem_undo_size(mem_undo_im_);
}


void line_init(linedata line, int x0, int y0, int x1, int y1)
{
	line[0] = x0;
	line[1] = y0;
	line[6] = line[8] = x1 < x0 ? -1 : 1;
	line[7] = line[9] = y1 < y0 ? -1 : 1;
	line[4] = abs(x1 - x0);
	line[5] = abs(y1 - y0);
	if (line[4] < line[5]) /* More vertical */
	{
		line[2] = line[3] = line[5];
		line[4] *= 2;
		line[5] *= 2;
		line[6] = 0;
	}
	else /* More horizontal */
	{
		line[2] = line[3] = line[4];
		line[4] = 2 * line[5];
		line[5] = 2 * line[2];
		line[7] = 0;
	}
}

int line_step(linedata line)
{
	line[3] -= line[4];
	if (line[3] <= 0)
	{
		line[3] += line[5];
		line[0] += line[8];
		line[1] += line[9];
	}
	else
	{
		line[0] += line[6];
		line[1] += line[7];
	}
	return (--line[2]);
}

void line_nudge(linedata line, int x, int y)
{
	while ((line[0] != x) && (line[1] != y) && (line[2] >= 0))
		line_step(line);
}

/* Produce a horizontal segment from two connected lines */
static void twoline_segment(int *xx, linedata line1, linedata line2)
{
	xx[0] = xx[1] = line1[0];
	while (TRUE)
	{
		if (!line1[7]) /* Segments longer than 1 pixel */
		{
			while ((line1[2] > 0) && (line1[3] > line1[4]))
				line_step(line1);
		}
		if (xx[0] > line1[0]) xx[0] = line1[0];
		if (xx[1] < line1[0]) xx[1] = line1[0];
		if ((line1[2] > 0) || (line2[2] < 0)) break;
		memcpy(line1, line2, sizeof(linedata));
		line2[2] = -1;
		if (xx[0] > line1[0]) xx[0] = line1[0];
		if (xx[1] < line1[0]) xx[1] = line1[0];
	}
}

void sline( int x1, int y1, int x2, int y2 )		// Draw single thickness straight line
{
	linedata line;

	line_init(line, x1, y1, x2, y2);
	for (; line[2] >= 0; line_step(line))
	{
		IF_IN_RANGE(line[0], line[1]) put_pixel(line[0], line[1]);
	}
}


void tline( int x1, int y1, int x2, int y2, int size )		// Draw size thickness straight line
{
	linedata line;
	int xv, yv;			// x/y vectors
	int xv2, yv2;
	int xdo, ydo, todo;
	float xuv, yuv, llen;		// x/y unit vectors, line length
	float xv1, yv1;			// vector for shadow x/y

	xdo = abs(x2 - x1);
	ydo = abs(y2 - y1);
	todo = xdo > ydo ? xdo : ydo;
	if (todo < 2) return;	// The 1st and last points are done by calling procedure

	if (size < 2) sline(x1, y1, x2, y2);
	/* Thick long line so use less accurate g_para */
	if ((size > 20) && (todo > 20))
	{
		xv = x2 - x1;
		yv = y2 - y1;
		llen = sqrt( xv * xv + yv * yv );
		xuv = ((float) xv) / llen;
		yuv = ((float) yv) / llen;

		xv1 = -yuv * ((float) size - 0.5);
		yv1 = xuv * ((float) size - 0.5);

		xv2 = mt_round(xv1 / 2 + 0.5*((size+1) %2) );
		yv2 = mt_round(yv1 / 2 + 0.5*((size+1) %2) );

		xv1 = -yuv * ((float) size - 0.5);
		yv1 = xuv * ((float) size - 0.5);

		g_para( x1 - xv2, y1 - yv2, x2 - xv2, y2 - yv2,
			mt_round(xv1), mt_round(yv1) );
	}
	/* Short or thin line so use more accurate but slower iterative method */
	else
	{
		line_init(line, x1, y1, x2, y2);
		for (line_step(line); line[2] > 0; line_step(line))
		{
			f_circle(line[0], line[1], size);
		}
	}
}

/* Draw whatever is bounded by two pairs of lines */
void draw_quad(linedata line1, linedata line2, linedata line3, linedata line4)
{
	int i, x1, x2, y1, xx[4];
	for (; line1[2] >= 0; line_step(line1) , line_step(line3))
	{
		y1 = line1[1];
		twoline_segment(xx + 0, line1, line2);
		twoline_segment(xx + 2, line3, line4);
		if ((y1 < 0) || (y1 >= mem_height)) continue;
		if (xx[0] > xx[2]) xx[0] = xx[2];
		if (xx[1] < xx[3]) xx[1] = xx[3];
		x1 = xx[0] < 0 ? 0 : xx[0];
		x2 = xx[1] >= mem_width ? mem_width - 1 : xx[1];
		for (i = x1; i <= x2; i++) put_pixel(i, y1);
	}
}

/* Draw general parallelogram */
void g_para( int x1, int y1, int x2, int y2, int xv, int yv )
{
	linedata line1, line2, line3, line4;
	int i, j, x[2] = {x1, x2}, y[2] = {y1, y2};

	j = (y1 < y2) ^ (yv < 0); i = j ^ 1;
	line_init(line1, x[i], y[i], x[i] + xv, y[i] + yv);
	line_init(line2, x[i] + xv, y[i] + yv, x[j] + xv, y[j] + yv);
	line_init(line3, x[i], y[i], x[j], y[j]);
	line_init(line4, x[j], y[j], x[j] + xv, y[j] + yv);
	draw_quad(line1, line2, line3, line4);
}


void f_rectangle( int x, int y, int w, int h )		// Draw a filled rectangle
{
	int i, j;

	if ( x<0 )
	{
		w = w + x;
		x = 0;
	}
	if ( y<0 )
	{
		h = h + y;
		y = 0;
	}
	if ( (x+w) > mem_width ) w = mem_width - x;
	if ( (y+h) > mem_height ) h = mem_height - y;

	for ( j=0; j<h; j++ )
	{
		for ( i=0; i<w; i++ ) put_pixel( x + i, y + j );
	}
}

void f_circle( int x, int y, int r )				// Draw a filled circle
{
	float r2, r3, k;
	int i, j, rx, ry, r4;
	int ox = x - r/2, oy = y - r/2;

	for ( j=0; j<r; j++ )
	{
		if ( r < 3 ) r4 = r-1;
		else
		{
			if ( r>10 ) k = 0.3 + 0.4*((float) j)/(r-1);	// Better for larger
			else k = 0.1 + 0.8*((float) j)/(r-1);		// Better for smaller

			r2 = 2*(k+(float) j) / r - 1;

			r3 = sqrt( 1 - r2*r2);
			if ( r%2 == 1 ) r4 = 2*mt_round( (r-1) * r3 / 2 );
			else
			{
				r4 = mt_round( (r-1) * r3 );
				if ( r4 % 2 == 0 ) r4--;
			}
		}
		ry = oy + j;
		rx = ox + (r-r4)/2;
		for ( i=0; i<=r4; i++)
		{
			IF_IN_RANGE( rx+i, ry ) put_pixel( rx+i, ry );
		}
	}
}


int mt_round( float n )			// Round a float to nearest whole number
{
	if ( n < 0 ) return ( (int) (n - 0.49999) );
	else return ( (int) (n + 0.49999) );
}

int get_next_line(char *input, int length, FILE *fp)
{
	char *st;

	st = fgets(input, length, fp);
	if ( st==NULL ) return -1;

	return 0;
}



/* This code assumes that source image is in bounds when enlarging */
/* Modes: 0 - clear, 1 - tile, 2 - mirror tile */
int mem_image_resize(int nw, int nh, int ox, int oy, int mode)
{
	chanlist old_img;
	char *src, *dest;
	int i, j, k, cc, bpp, oxo = 0, oyo = 0, nxo = 0, nyo = 0, ow, oh, res;
	int oww = mem_width, ohh = mem_height, mirr = 0, tw = mem_width;

	nw = nw < 1 ? 1 : nw > MAX_WIDTH ? MAX_WIDTH : nw;
	nh = nh < 1 ? 1 : nh > MAX_HEIGHT ? MAX_HEIGHT : nh;
	if ((nw <= oww) && (nh <= ohh)) mode = 0;

	memcpy(old_img, mem_img, sizeof(chanlist));
	res = undo_next_core(UC_NOCOPY, nw, nh, mem_img_bpp, CMASK_ALL);
	if (res) return 1;			// Not enough memory

	if ( ox < 0 ) oxo = -ox;
	else nxo = ox;
	if ( oy < 0 ) oyo = -oy;
	else nyo = oy;

	if (!mode) /* Clear */
	{
		j = nw * nh;
		for (cc = 0; cc < NUM_CHANNELS; cc++)
		{
			if (!mem_img[cc]) continue;
			dest = mem_img[cc];
			if ((cc != CHN_IMAGE) || (mem_img_bpp == 1))
			{
				memset(dest, cc == CHN_IMAGE ? mem_col_A : 0, j);
				continue;
			}
			for (i = 0; i < j; i++)	// Background is current colour A
			{
				*dest++ = mem_col_A24.red;
				*dest++ = mem_col_A24.green;
				*dest++ = mem_col_A24.blue;
			}
		}
		ow = oww < nw ? oww : nw;
	}
	else /* Tile - prepare for horizontal pass */
	{
		mirr = (mode == 2) && (oww > 2) ? 1 : 0;
		ow = nw;
		tw -= mirr;
		i = (nxo + tw - 1) / tw;
		if (i & 1) mirr = -mirr;
		oxo = i * tw - nxo;
		nxo = 0;
	}
	oh = ohh < nh ? ohh : nh;

	/* Do horizontal tiling */
	for (; ow; ow -= res)
	{	
		res = tw - oxo < ow ? tw - oxo : ow;
		for (cc = 0; cc < NUM_CHANNELS; cc++)
		{
			if (!mem_img[cc]) continue;
			bpp = BPP(cc);
			j = res * bpp;
			for (i = 0; i < oh; i++)
			{
				src = old_img[cc] + (oxo + oww * (i + oyo)) * bpp;
				dest = mem_img[cc] + (nxo + nw * (i + nyo)) * bpp;
				/* Normal copy */
				if (mirr >= 0)
				{
					memcpy(dest, src, j);
					continue;
				}
				/* Reverse copy */
				src += (oww - oxo - oxo - 1) * bpp;
				for (k = 0; k < j; k += bpp , src -= bpp)
				{
					dest[k] = src[0];
					if (bpp == 1) continue;
					dest[k + 1] = src[1];
					dest[k + 2] = src[2];
				}
			}
		}
		nxo += res;
		oxo = 0;
		mirr = -mirr;
	}

	/* Only one stripe? */
	if (!mode || (nh <= ohh)) return (0);

	/* Tile up & down */
	if (ohh < 3) mirr = 0;
	for (cc = 0; cc < NUM_CHANNELS; cc++)
	{
		if (!mem_img[cc]) continue;
		bpp = nw * BPP(cc);
		i = nyo - 1;
		dest = mem_img[cc] + i * bpp;
		if (mirr) /* Reverse copy up */
		{
			j = i - (ohh - 2);
			if (j < -1) j = -1;
			src = dest + 2 * bpp;
			for (; i > j; i--)
			{
				memcpy(dest, src, bpp);
				dest -= bpp;
				src += bpp;
			}
		}
		/* Forward copy up */
		src = mem_img[cc] + (nyo + ohh - 1) * bpp;
		for (; i >= 0; i--)
		{
			memcpy(dest, src, bpp);
			dest -= bpp;
			src -= bpp;
		}
		i = nyo + ohh;
		dest = mem_img[cc] + i * bpp;
		if (mirr) /* Reverse copy down */
		{
			j = i + ohh - 2;
			if (j > nh) j = nh;
			src = dest - 2 * bpp;
			for (; i < j; i++)
			{
				memcpy(dest, src, bpp);
				dest += bpp;
				src -= bpp;
			}
		}
		/* Forward copy down */
		src = mem_img[cc] + nyo * bpp;
		for (; i < nh; i++)
		{
			memcpy(dest, src, bpp);
			dest += bpp;
			src += bpp;
		}
	}

	return 0;
}

int get_pixel( int x, int y )	/* Mixed */
{
	x = mem_width * y + x;
	if ((mem_channel != CHN_IMAGE) || (mem_img_bpp == 1))
		return (mem_img[mem_channel][x]);
	x *= 3;
	return (MEM_2_INT(mem_img[CHN_IMAGE], x));
}

int get_pixel_RGB( int x, int y )	/* RGB */
{
	x = mem_width * y + x;
	if (mem_img_bpp == 1)
		return (PNG_2_INT(mem_pal[mem_img[CHN_IMAGE][x]]));
	x *= 3;
	return (MEM_2_INT(mem_img[CHN_IMAGE], x));
}

int get_pixel_img( int x, int y )	/* RGB or indexed */
{
	x = mem_width * y + x;
	if (mem_img_bpp == 1) return (mem_img[CHN_IMAGE][x]);
	x *= 3;
	return (MEM_2_INT(mem_img[CHN_IMAGE], x));
}

void prep_mask(int start, int step, int cnt, unsigned char *mask,
	unsigned char *mask0, unsigned char *img0)
{
	cnt = start + step * (cnt - 1) + 1;

	if (mask0) memcpy(mask, mask0, cnt);
	else memset(mask, 0, cnt);
}

void row_protected(int x, int y, int len, unsigned char *mask)
{
	unsigned char *mask0 = NULL;
	int ofs = y * mem_width + x;

	prep_mask(0, 1, len, mask, mask0, mem_img[CHN_IMAGE] + ofs * mem_img_bpp);
}

void put_pixel( int x, int y )
{
	int offset;

	offset = 3*(x + mem_width * y);

	mem_img[CHN_IMAGE][offset]     = mem_col_A24.red;
	mem_img[CHN_IMAGE][offset + 1] = mem_col_A24.green;
	mem_img[CHN_IMAGE][offset + 2] = mem_col_A24.blue;
}

void process_mask(int start, int step, int cnt, unsigned char *mask,
	unsigned char *alphar, unsigned char *alpha0, unsigned char *alpha,
	unsigned char *trans, int opacity, int noalpha)
{
	unsigned char newc, oldc;
	int i, j, k;

	cnt = start + step * cnt;

	/* Opacity mode */
	if (opacity)
	{
		for (i = start; i < cnt; i += step)
		{
			k = (255 - mask[i]) * opacity;
			if (!k)
			{
				mask[i] = 0;
				continue;
			}
			k = (k + (k >> 8) + 1) >> 8;

			if (trans)
			{
				/* Have transparency mask */
				k *= trans[i];
				k = (k + (k >> 8) + 1) >> 8;
			}
			mask[i] = k;

			if (!alpha || !k) continue;
			/* Have alpha channel - process it */
			newc = alpha[i];
			oldc = alpha0[i];
			j = oldc * 255 + (newc - oldc) * k;
			alphar[i] = (j + (j >> 8) + 1) >> 8;
			if (noalpha) continue;
			if (j) mask[i] = (255 * k * newc) / j;
		}
	}

	/* Indexed mode with transparency mask and/or alpha */
	else if (trans || alpha)
	{
		for (i = start; i < cnt; i += step)
		{
			if (trans) mask[i] |= trans[i] ^ 255;
			if (!alpha || mask[i]) continue;
			/* Have alpha channel - process it */
			newc = alpha[i];
			alphar[i] = newc;
		}
	}
}

void process_img(int start, int step, int cnt, unsigned char *mask,
	unsigned char *imgr, unsigned char *img0, unsigned char *img,
	int opacity, int sourcebpp)
{
	unsigned char newc;//, oldc;
	unsigned char r, g, b, nr, ng, nb;
	int i, j, ofs3;

	cnt = start + step * cnt;

	/* Indexed image or utility channel */
	if (!opacity)
	{
		for (i = start; i < cnt; i += step)
		{
			if (mask[i]) continue;
			newc = img[i];
			imgr[i] = newc;
		}
	}

	/* RGB image */
	else
	{
		for (i = start; i < cnt; i += step)
		{
			opacity = mask[i];
			if (!opacity) continue;
			ofs3 = i * 3;
			if (sourcebpp == 3) /* RGB-to-RGB paste */
			{
				nr = img[ofs3 + 0];
				ng = img[ofs3 + 1];
				nb = img[ofs3 + 2];
			}
			else /* Indexed-to-RGB paste */
			{
				nr = mem_pal[img[i]].red;
				ng = mem_pal[img[i]].green;
				nb = mem_pal[img[i]].blue;
			}
			if (opacity == 255)
			{
				imgr[ofs3 + 0] = nr;
				imgr[ofs3 + 1] = ng;
				imgr[ofs3 + 2] = nb;
				continue;
			}
			r = img0[ofs3 + 0];
			g = img0[ofs3 + 1];
			b = img0[ofs3 + 2];
			j = r * 255 + (nr - r) * opacity;
			imgr[ofs3 + 0] = (j + (j >> 8) + 1) >> 8;
			j = g * 255 + (ng - g) * opacity;
			imgr[ofs3 + 1] = (j + (j >> 8) + 1) >> 8;
			j = b * 255 + (nb - b) * opacity;
			imgr[ofs3 + 2] = (j + (j >> 8) + 1) >> 8;
		}
	}	
}

/* Separate function for faster paste */
void paste_pixels(int x, int y, int len, unsigned char *mask, unsigned char *img,
	unsigned char *alpha, unsigned char *trans, int opacity)
{
	unsigned char *old_image, *old_alpha = NULL, *dest = NULL;
	int bpp, ofs = x + mem_width * y;

	bpp = MEM_BPP;

	process_mask(0, 1, len, mask, dest, old_alpha, NULL, trans, opacity, 0);

	/* Stop if we have alpha without image */
	if (!img) return;

	old_image = mem_img[mem_channel];
	old_image += ofs * bpp;
	dest = mem_img[mem_channel] + ofs * bpp;

	process_img(0, 1, len, mask, dest, old_image, img, opacity, mem_clip_bpp);
}


///	CLIPBOARD MASK

int mem_clip_mask_init(unsigned char val)		// Initialise the clipboard mask
{
	int j = mem_clip_w*mem_clip_h;

	if ( mem_clipboard != NULL ) mem_clip_mask_clear();	// Remove old mask

	mem_clip_mask = malloc(j);
	if (!mem_clip_mask) return 1;			// Not able to allocate memory

	memset(mem_clip_mask, val, j);		// Start with fully opaque/clear mask

	return 0;
}

void mem_clip_mask_clear()		// Clear/remove the clipboard mask
{
	if ( mem_clip_mask != NULL )
	{
		free(mem_clip_mask);
		mem_clip_mask = NULL;
	}
}





/*
 * This flood fill algorithm processes image in quadtree order, and thus has
 * guaranteed upper bound on memory consumption, of order O(width + height).
 * (C) Dmitry Groshev
 */
#define QLEVELS 11
#define QMINSIZE 32
#define QMINLEVEL 5
void wjfloodfill(int x, int y, int col, unsigned char *bmap, int lw)
{
	short *nearq, *farq;
	int qtail[QLEVELS + 1], ntail = 0;
	int borders[4] = {0, mem_width, 0, mem_height};
	int corners[4], levels[4], coords[4];
	int i, j, k, kk, lvl, tx, ty, imgc = 0, fmode = 0, /*lastr[3],*/ thisr[3];
	int bidx = 0, bbit = 0;
//	double lastc[3], thisc[3], dist2, mdist2 = flood_step * flood_step;
//	csel_info *flood_data = NULL;
	char *tmp = NULL;

	/* Init */
	if ((x < 0) || (x >= mem_width) || (y < 0) || (y >= mem_height) ||
		(get_pixel(x, y) != col) /*|| (pixel_protected(x, y) == 255)*/)
		return;
	i = ((mem_width + mem_height) * 3 + QMINSIZE * QMINSIZE) * 2 * sizeof(short);
	nearq = malloc(i); // Exact limit is less, but it's too complicated 
	if (!nearq) return;
	farq = nearq + QMINSIZE * QMINSIZE;
	memset(qtail, 0, sizeof(qtail));

	/* Start drawing */
	if (bmap) bmap[y * lw + (x >> 3)] |= 1 << (x & 7);
	else
	{
		put_pixel(x, y);
		if (get_pixel(x, y) == col)
		{
			/* Can't draw */
			free(nearq);
			return;
		}
	}

/*
	// Configure fuzzy flood fill
	if (flood_step && ((mem_channel == CHN_IMAGE) || flood_img))
	{
		if (flood_slide) fmode = flood_cube ? 2 : 3;
		else flood_data = ALIGNTO(tmp = calloc(1, sizeof(csel_info)
			+ sizeof(double)), double);
		if (flood_data)
		{
			flood_data->center = get_pixel_RGB(x, y);
			flood_data->range = flood_step;
			flood_data->mode = flood_cube ? 2 : 0;
			csel_reset(flood_data);
			fmode = 1;
		}
	}
	// Configure by-image flood fill
	else*/
	if (/*!flood_step && flood_img &&*/ (mem_channel != CHN_IMAGE))
	{
		imgc = get_pixel_img(x, y);
		fmode = -1;
	}

	while (1)
	{
		/* Determine area */
		corners[0] = x & ~(QMINSIZE - 1);
		corners[1] = corners[0] + QMINSIZE;
		corners[2] = y & ~(QMINSIZE - 1);
		corners[3] = corners[2] + QMINSIZE;
		/* Determine queue levels */
		for (i = 0; i < 4; i++)
		{
			j = (corners[i] & ~(corners[i] - 1)) - 1;
			j = (j & 0x5555) + ((j & 0xAAAA) >> 1);
			j = (j & 0x3333) + ((j & 0xCCCC) >> 2);
			j = (j & 0x0F0F) + ((j & 0xF0F0) >> 4);
			levels[i] = (j & 0xFF) + (j >> 8) - QMINLEVEL;
		}
		/* Process near points */
		while (1)
		{
			coords[0] = x;
			coords[2] = y;
			if (fmode > 1)
			{
				k = get_pixel_RGB(x, y);
/*				if (fmode == 3) get_lxn(lastc, k);
				else
				{
					lastr[0] = INT_2_R(k);
					lastr[1] = INT_2_G(k);
					lastr[2] = INT_2_B(k);
				}*/
			}
			for (i = 0; i < 4; i++)
			{
				coords[1] = x;
				coords[3] = y;
				coords[(i & 2) + 1] += ((i + i) & 2) - 1;
				/* Is pixel valid? */
				if (coords[i] == borders[i]) continue;
				tx = coords[1];
				ty = coords[3];
				if (bmap)
				{
					bidx = ty * lw + (tx >> 3);
					bbit = 1 << (tx & 7);
					if (bmap[bidx] & bbit) continue;
				}
				/* Sliding mode */
				switch (fmode)
				{
/*				case 3: // Sliding L*X*N*
					get_lxn(thisc, get_pixel_RGB(tx, ty));
					dist2 = (thisc[0] - lastc[0]) * (thisc[0] - lastc[0]) +
						(thisc[1] - lastc[1]) * (thisc[1] - lastc[1]) +
						(thisc[2] - lastc[2]) * (thisc[2] - lastc[2]);
					if (dist2 > mdist2) continue;
					break;*/
				case 2: // Sliding RGB
					k = get_pixel_RGB(tx, ty);
					thisr[0] = INT_2_R(k);
					thisr[1] = INT_2_G(k);
					thisr[2] = INT_2_B(k);
/*					if ((abs(thisr[0] - lastr[0]) > flood_step) ||
						(abs(thisr[1] - lastr[1]) > flood_step) ||
						(abs(thisr[2] - lastr[2]) > flood_step))
						continue;*/
					break;
/*				case 1: // Centered mode
					if (!csel_scan(ty * mem_width + tx, 1, 1,
						NULL, mem_img[CHN_IMAGE], flood_data))
						continue;
					break;*/
				case 0: /* Normal mode */
					if (get_pixel(tx, ty) != col) continue;
					break;
				default: /* (-1) - By-image mode */
					if (get_pixel_img(tx, ty) != imgc) continue;
					break;
				}
				/* Is pixel writable? */
				if (bmap)
				{
//					if (pixel_protected(tx, ty) == 255)
//						continue;
					bmap[bidx] |= bbit;
				}
				else
				{
					put_pixel(tx, ty);
					if (get_pixel(tx, ty) == col) continue;
				}
				/* Near queue */
				if (coords[i] != corners[i])
				{
					nearq[ntail++] = tx;
					nearq[ntail++] = ty;
					continue;
				}
				/* Far queue */
				lvl = levels[i];
				for (j = 0; j < lvl; j++) // Slide lower levels
				{
					k = qtail[j];
					qtail[j] = k + 2;
					if (k > qtail[j + 1])
					{
						kk = qtail[j + 1];
						farq[k] = farq[kk];
						farq[k + 1] = farq[kk + 1];
					}
				}
				k = qtail[lvl];
				farq[k] = tx;
				farq[k + 1] = ty;
				qtail[lvl] = k + 2;
			}
			if (!ntail) break;
			y = nearq[--ntail];
			x = nearq[--ntail];
		}
		/* All done? */
		if (!qtail[0]) break;
		i = qtail[0] - 2;
		x = farq[i];
		y = farq[i + 1];
		qtail[0] = i;
		for (j = 1; qtail[j] > i; j++)
			qtail[j] = i;
	}
	free(nearq);
	free(tmp);
}

/* Flood fill - may use temporary area (1 bit per pixel) */
void flood_fill(int x, int y, unsigned int target)
{
//	unsigned char *pat, *temp;
//	int i, j, k, lw = (mem_width + 7) >> 3;

	/* Regular fill? */
//	if (!tool_pat && (tool_opacity == 255) && !flood_step &&
//		(!flood_img || (mem_channel == CHN_IMAGE)))
	{
		wjfloodfill(x, y, target, NULL, 0);
		return;
	}

/*
	j = lw * mem_height;
	pat = temp = malloc(j);
	if (!pat)
	{
		memory_errors(1);
		return;
	}
	memset(pat, 0, j);
	wjfloodfill(x, y, target, pat, lw);
	for (i = 0; i < mem_height; i++)
	{
		for (j = 0; j < mem_width; )
		{
			k = *temp++;
			if (!k)
			{
				j += 8;
				continue;
			}
			for (; k; k >>= 1)
			{
				if (k & 1) put_pixel(j, i);
				j++;
			}
			j = (j + 7) & ~(7);
		}
	}
	free(pat);*/
}
