///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "2D_D.h"
#include "rheolef/tensor4.h"
#include "rheolef/ublas_matrix_range.h"
#include "map_projector.h"

namespace rheolef {
using namespace std;
using namespace ublas;

void
compute_tensor_c (
    const geo_element& K, 
    const tensor&      G, 
    const tensor&      P, 
    size_t             d,
    size_t             map_d,
    tensor4&           C)
{
 typedef _2D_D::size_type size_type;
 C = 0;
 if (d > map_d) {
   // ------------------------------------------
   // cas surfacique
   // ------------------------------------------
   // calcul de B 
   tensor4 B;
   for (size_type i = 0; i < d; i++)
   for (size_type j = 0; j < d; j++) {
       for (size_type k = 0; k < d;     k++)
       for (size_type r = 0; r < map_d; r++) {
           Float sum = 0;
           for (size_type l = 0; l < d; l++) {
	      sum += (P(i,k)*P(l,j) + P(i,l)*P(k,j))*G(l,r);
           }
	   B(i,j,k,r) = sum;
      	}
   }
   // calcul de C cas surfacique  
   for (size_type k = 0; k < d;     k++)
   for (size_type r = 0; r < map_d; r++) {
       for (size_type p = 0; p < d;     p++)
       for (size_type t = 0; t < map_d; t++) {
           for (size_type i = 0; i < d; i++)
           for (size_type j = 0; j < d; j++) {
	       C(i,j,k,r) += B(i,j,k,r)*B(i,j,p,t);
           }
       }
   }
 } else {
   // ------------------------------------------
   //  calcul de C cas classique;
   // ------------------------------------------
   for (size_type r = 0; r < map_d; r++) 
   for (size_type t = 0; t < map_d; t++) {
     for (size_type k = 0; k < d; k++) {
       Float sum = 0;
       for (size_type i = 0; i < d; i++) {
         sum += 2*G(i,r)*G(i,t);
       }
       C(k,r,k,t) += sum;
     }
     for (size_type k = 0; k < d; k++)
     for (size_type p = 0; p < d; p++) {
       for (size_type i = 0; i < d; i++) {
         C(k,r,p,t) += 2*G(p,r)*G(k,t);
       }
     }
   }
 }
}
void
_2D_D::operator() (const geo_element& K, matrix<Float>& ak) const
{
  size_type nj = get_first_basis().size(K);
  size_type ni = get_second_basis().size(K);
#undef USE_NEW_VERSION
#ifdef USE_NEW_VERSION
  size_type d     = coordinate_dimension();
  size_type map_d = K.dimension();
  reference_element hat_K = K;
  size_type nx = get_first_basis().size(hat_K);
  size_type ny = get_second_basis().size(hat_K);
  ak.resize (d*ny, d*nx);
  ak.clear();
  quadrature::const_iterator first = _quad.begin(hat_K);
  quadrature::const_iterator last  = _quad.end  (hat_K);
  tensor DF;
  tensor invDF;
  tensor G;
  tensor P; // P = I - nxn
  tensor4 C;
  for (size_type q = 0; first != last; first++, q++) {
    jacobian_piola_transformation (K,q,DF);
    invDF = pseudo_inverse_jacobian_piola_transformation (DF, map_d);
    Float wq = is_weighted() ? weight(K,q) : Float(1);
    wq *= weight_coordinate_system (K, q);
    wq *= det_jacobian_piola_transformation (DF, K.dimension());
    wq *= (*first).w;
    jacobian_piola_transformation (K,q,DF);
    G = trans(invDF);
    map_projector (DF, d, map_d, P);
    compute_tensor_c (K, G, P, d, map_d, C);
    basis_on_quadrature::const_iterator_grad grad_phi       = _by_table.begin_grad(hat_K, q);
    basis_on_quadrature::const_iterator_grad last_grad_phi  = _by_table.end_grad(hat_K, q);
    basis_on_quadrature::const_iterator_grad first_grad_psi = _bx_table.begin_grad (hat_K, q);
    basis_on_quadrature::const_iterator_grad last_grad_psi  = _bx_table.end_grad (hat_K, q);
    for (size_type i = 0; grad_phi != last_grad_phi; grad_phi++, i++) {
      const point& grad_phi_i = *grad_phi;
      basis_on_quadrature::const_iterator_grad grad_psi = first_grad_psi;
      for (size_type j = 0; grad_psi != last_grad_psi; grad_psi++, j++) {
        const point& grad_psi_j = *grad_psi;
        for (size_type k = 0; k < d; k++)
        for (size_type p = 0; p < d; p++)
        for (size_type r = 0; r < map_d; r++)
        for (size_type t = 0; t < map_d; t++) {
	  ak(i+k*ny,j+p*nx) += C(k,r,p,t)*wq*grad_phi_i[r]*grad_psi_j[t];
	}
      }
    }
  }
#else // USE_OLD_VERSION
  build_gradt_grad (K, ak);
  switch (K.dimension()) {
    case 1:
      ak *= 2;
      break;
    case 2:
    case 3: {
      matrix<Float> trace (ni, nj);
      trace.clear();
      for (size_type k = 0; k < K.dimension(); k++) {
        matrix_range<matrix<Float> > dk (ak, range(k*ni,(k+1)*ni),range(k*nj,(k+1)*nj));
        trace += dk;
      }
      for (size_type k = 0; k < K.dimension(); k++) {
        matrix_range<matrix<Float> > dk (ak, range(k*ni,(k+1)*ni), range(k*nj,(k+1)*nj));
        dk += trace;
     }
     break;
    }
    default: {
      fatal_macro ("unexpected element `" << K.name() << "'");
      break;
    }
  }
#endif // USE_OLD_VERSION
  if (coordinate_system_type() == fem_helper::cartesian) return;
  // --------------------------------------
  // otherwise, add the (ur/r) (vr/r) r dr dz term
  //   as : ur vr (1/r) dr dz
  // i.e. using the dual `1/r' extra weight
  // --------------------------------------
  ublas::matrix<Float> m_invr;
  set_use_coordinate_system_dual_weight (true);
  build_scalar_mass (K, m_invr);
  set_use_coordinate_system_dual_weight (false);
  size_type k = 0;
  if (coordinate_system_type() == fem_helper::axisymmetric_zr) k++;
  matrix_range<matrix<Float> > dk (ak, range(k*ni,(k+1)*ni), range(k*nj,(k+1)*nj));
  dk += 2.*m_invr;
}

_2D_D::size_type
_2D_D::n_derivative() const
{
  if (coordinate_system_type() == fem_helper::cartesian) return 2;
  else return 0;
}

void
_2D_D::check_after_initialize () const
{
  size_type d = coordinate_dimension();
  check_macro (
	d == get_first_space().n_component() && 
	d == get_second_space().n_component(),
	"unsupported non-vectorial space for `2D_D' form");
}

} // namespace rheolef
