(* VDU routines for an AED512 terminal. *)

#include 'globals.h';
#include 'screenio.h';
#include 'vdu.h';

CONST
   screenht = 512;       (* change to 483 for smaller AED screen *)

   ALPHA = CHR(1b);      (* SOH = enter Alpha mode *)
   INTER = CHR(33b);     (* ESC = enter Interpreter mode *)
   SAP = '^';            (* select Alpha parameters for char size *)
   MOV = 'Q';            (* move CAP to absolute x,y position;
                            CAP = Current Access Position *)
   DVA = 'A';            (* draw vector from CAP to absolute x,y position *)
   DFR = 'o';            (* draw filled rectangle with one corner at CAP and
                            opposite corner at absolute x,y position *)
   SEC = 'C';            (* set current writing colour *)
   SBC = '[';            (* set background colour *)
   green = CHR(2b);      (* my writing colour; better than default red *)
   black = CHR(0b);      (* my background colour; also the default *)
   lineht = 9;           (* height in pixels of my default text line *)
   charwidth1 = 6;       (* 5*8 font for dialogue region; 85 chars per line *)
   charwidth2 = 8;       (* normal 7*12 font; 64 chars per line *)
   charwidth3 = 12;      (* double 5*8 font; 42 chars per line *)
   charwidth4 = 16;      (* double 7*12 font; 32 chars per line *)

VAR
   maxy,                  (* screenht - 1 *)
   loadedwidth,           (* remember charwidth set by last LoadFont *)
   charwidth : INTEGER;   (* used to select Alpha font *)

(******************************************************************************)

PROCEDURE StartText;

(* We are about to draw text in dialogue region. *)

BEGIN
WriteChar(ALPHA);
END; (* StartText *)

(******************************************************************************)

PROCEDURE OutputXY (x, y : INTEGER);

(* Assuming non-mnemonic 8-bit encoding (the default),
   send the 3 bytes representing the given x,y address.
*)

BEGIN
WriteChar(CHR((x DIV 256) * 16 + (y DIV 256)));(* x11,x10,x9,x8,y11,y10,y9,y8 *)
WriteChar(CHR(x MOD 256));                     (* x7 ,x6 ,   ...          ,x0 *)
WriteChar(CHR(y MOD 256));                     (* y7 ,y6 ,   ...          ,y0 *)
END; (* OutputXY *)

(******************************************************************************)

PROCEDURE SelectAlphaFont;

(* Select a new Alpha font depending on the current value of charwidth
   and prepare to display Alpha text.
   charwidth is charwidth1 initially; changed in MoveToTextLine and LoadFont.
*)

BEGIN
WriteChar(INTER);
WriteChar(SAP);    (* select Alpha parameters *)
CASE charwidth OF
charwidth1 :
      BEGIN
      WriteChar('1');               (* normal sized *)
      WriteChar('5');               (* 5*8 font *)
      WriteChar(CHR(charwidth1));   (* default hspace *)
      WriteChar(CHR(lineht));       (* default vspace *)
      END;
charwidth2 :
      BEGIN
      WriteChar('1');               (* normal sized *)
      WriteChar('7');               (* 7*12 font *)
      WriteChar(CHR(charwidth2));
      WriteChar(CHR(15b));          (* vspace = 13 *)
      END;
charwidth3 :
      BEGIN
      WriteChar('2');               (* double sized *)
      WriteChar('5');               (* 5*8 font *)
      WriteChar(CHR(charwidth3));
      WriteChar(CHR(22b));          (* vspace = 18 *)
      END;
charwidth4 :
      BEGIN
      WriteChar('2');               (* double sized *)
      WriteChar('7');               (* 7*12 font *)
      WriteChar(CHR(charwidth4));
      WriteChar(CHR(32b));          (* vspace = 26 *)
      END;
OTHERWISE
      WriteString('Bug in AED512VDU! Illegal charwidth!');
END;
WriteChar('L');    (* Alpha cursor position identical with CAP *)
WriteChar(ALPHA);
END; (* SelectAlphaFont *)

(******************************************************************************)

PROCEDURE MoveToTextLine (line : INTEGER);

(* Move to start of given text line; at the end of this routine we must
   be ready to display default Alpha characters.
*)

BEGIN
WriteChar(INTER);
WriteChar(MOV);
OutputXY(0,maxy-(line*lineht));
charwidth := charwidth1;   (* use default Alpha font *)
SelectAlphaFont;
END; (* MoveToTextLine *)

(******************************************************************************)

PROCEDURE ClearTextLine (line : INTEGER);

(* Erase given line; note that DVItoVDU does not assume anything about the
   current position at the end of this routine.
*)

BEGIN
WriteChar(INTER);
WriteChar(SEC);
WriteChar(black);   (* set writing colour to background temporarily *)
WriteChar(MOV);
OutputXY(0,                     (* set CAP to bottom left corner of text line *)
         maxy-(line*lineht)-2); (* ensure g,j,p... descenders are erased *)
WriteChar(DFR);     (* draw filled rectangle to top right corner *)
OutputXY(511,maxy-(line*lineht)+(lineht-3));
WriteChar(SEC);
WriteChar(green);   (* back to green *)
WriteChar(ALPHA);   (* safer to leave in Alpha mode *)
END; (* ClearTextLine *)

(******************************************************************************)

PROCEDURE ClearScreen;

BEGIN
WriteChar(INTER);
WriteChar(FF);      (* erase screen and return cursor to home position;
                       also exits Interpreter *)
(* preferable to ERS = '~' which has some problems with setting x,y origin *)
END; (* ClearScreen *)

(******************************************************************************)

PROCEDURE StartGraphics;

(* We are about to draw in window region. *)

BEGIN
IF charwidth <> loadedwidth THEN BEGIN   (* graphics mode was interrupted *)
   charwidth := loadedwidth;
   SelectAlphaFont;                (* restore font selected in last LoadFont *)
END;
WriteChar(INTER);
END; (* StartGraphics *)

(******************************************************************************)

PROCEDURE LoadFont (fontname : string;
                    fontsize : INTEGER;
                    mag, hscale, vscale : REAL);

(* AED512 uses the given information to select an appropriate character size
   (based on horizontal scaling only!) for future calls of ShowChar.
   DVItoVDU only calls this routine in graphics mode.
*)

VAR newwidth : INTEGER;

BEGIN
(* convert fontsize into scaled screen pixels using mag and hscale *)
fontsize := TRUNC( (fontsize * mag * hscale) + 0.5 );
(* Chooose one of the 4 Alpha mode character sizes based on fontsize:
   charwidth    max chars/line      fontsize range
       1             85                0..20
       2             64               21..40
       3             42               41..60
       4             32               61...
   The fontsize ranges were chosen by trial and error.
*)
IF      fontsize < 21 THEN
   newwidth := charwidth1
ELSE IF fontsize < 41 THEN
   newwidth := charwidth2
ELSE IF fontsize < 61 THEN
   newwidth := charwidth3
ELSE
   newwidth := charwidth4;
loadedwidth := newwidth;              (* remember for ShowChar *)
IF charwidth <> newwidth THEN BEGIN   (* change needed *)
   charwidth := newwidth;
   SelectAlphaFont;
END;
WriteChar(INTER);                     (* must exit in graphics mode *)
END; (* LoadFont *)

(******************************************************************************)

PROCEDURE ShowChar (screenh, screenv : INTEGER;
                    ch : CHAR);

(* We show the given character (mapping TeX text character to ASCII) using
   the given reference point and the charwidth selected by the last LoadFont.
   DVItoVDU only uses this routine in a Terse display.
   Note that Alpha characters g,j,p,q,y have descenders that fall below their
   reference point; when baselines appear near the bottom of the screen the
   descenders will wrap to the top.
*)

VAR newch : CHAR;   (* = TeXtoASCII[ch] *)

BEGIN
(* shift character left if it will overlap right edge of screen *)
IF screenh + charwidth > 511 THEN
   screenh := 511 - charwidth;
(* DVItoVDU only calls ShowChar in graphics mode *)
WriteChar(MOV);
OutputXY(screenh,maxy-screenv);    (* move cursor to reference point *)

(* We use TeXtoASCII to map ch into a comparable ASCII character, apart
   from most of the ? characters which we attempt to simulate.
*)

WriteChar(ALPHA);
newch := TeXtoASCII[ch];
IF newch <> '?' THEN
   (* newch is similar to TeX ch *)
   WriteChar(newch)
ELSE
   (* attempt to display something other than ? *)
   CASE ORD(ch) OF
   13b..17b :   (* ff, fi, fl, ffi, ffl *)
       BEGIN
       WriteChar('f');
       (* only simulate rest of ligature if room at right edge *)
       IF screenh + 2 * charwidth - (charwidth DIV 2) <= 511 THEN BEGIN
          WriteChar(INTER);
          WriteChar(MOV);
          OutputXY(screenh + charwidth - (charwidth DIV 2),maxy-screenv);
          WriteChar(ALPHA);
          CASE ORD(ch) OF
          13b : WriteChar('f') ;
          14b : WriteChar('i') ;
          15b : WriteChar('l') ;
          16b,
          17b : BEGIN
                WriteChar('f');
                IF screenh + 3 * charwidth - 2 * (charwidth DIV 2) <= 511 THEN
                   BEGIN
                   WriteChar(INTER);
                   WriteChar(MOV);
                   OutputXY(screenh + 2 * charwidth - 2 * (charwidth DIV 2),
                            maxy-screenv);
                   WriteChar(ALPHA);
                   IF ch = CHR(16b) THEN
                      WriteChar('i')
                   ELSE
                      WriteChar('l');
                END;
                END;
          END;
       END;
       END;
   31b : WriteChar('B');   (* German sharp S *)
   32b, 33b, 35b, 36b :    (* diphthongs: ae, oe, AE, OE *)
       BEGIN
       CASE ORD(ch) OF
       32b : WriteChar('a') ;
       33b : WriteChar('o') ;
       35b : WriteChar('A') ;
       36b : WriteChar('O')
       END;
       IF screenh + 2 * charwidth - (charwidth DIV 2) <= 511 THEN BEGIN
          WriteChar(INTER);
          WriteChar(MOV);
          OutputXY(screenh + charwidth - (charwidth DIV 2),maxy-screenv);
          WriteChar(ALPHA);
          CASE ORD(ch) OF
          32b, 33b : WriteChar('e') ;
          35b, 36b : WriteChar('E')
          END;
       END;
       END;
   40b : WriteChar('''');   (* Polish suppressed l and L *)
   OTHERWISE
       WriteChar('?');
   END;
WriteChar(INTER);   (* must exit in graphics mode *)
END; (* ShowChar *)

(******************************************************************************)

PROCEDURE ShowRectangle (screenh, screenv,          (* top left pixel *)
                         width, height : INTEGER;   (* size of rectangle *)
                         ch : CHAR);                (* black pixel *)

(* Display the given rectangle without using the given black pixel character.
   We assume that the top left position is correct and that the given
   dimensions do not go beyond the screen edges.
   We assume DVItoVDU is in graphics mode.
*)

VAR pos : INTEGER;

BEGIN
IF height = 1 THEN BEGIN                    (* show row vector *)
   pos := maxy-screenv;
   WriteChar(MOV);
   OutputXY(screenh,pos);                   (* move cursor to start of row *)
   WriteChar(DVA);
   OutputXY(screenh+width-1,pos);           (* draw vector to end of row *)
END
ELSE IF width = 1 THEN BEGIN                (* show column vector *)
   WriteChar(MOV);
   OutputXY(screenh,maxy-screenv);          (* move cursor to start of column *)
   WriteChar(DVA);
   OutputXY(screenh,maxy-(screenv+height-1)); (* draw vector to end of column *)
END
ELSE BEGIN                                  (* draw filled rectangle *)
   (* assume height and width > 1 *)
   WriteChar(MOV);
   OutputXY(screenh,maxy-screenv);          (* CAP to top left corner *)
   WriteChar(DFR);
   OutputXY(screenh+width-1,
            maxy-(screenv+height-1));       (* fill to bottom right corner *)
END;
END; (* ShowRectangle *)

(******************************************************************************)

PROCEDURE ResetVDU;

(* We DON'T reset AED512 to power-up state; we leave AED using green as the
   writing colour and set font back to power-up values.
   Such decisions may not please other AED sites, but I was talked into them
   by our AED users (all 2 of them!).
*)

BEGIN
WriteChar(INTER);
WriteChar(SAP);          (* select Alpha parameters *)
WriteChar('1');          (* normal sized *)
WriteChar('7');          (* 7*12 font *)
WriteChar(CHR(8));       (* hspace = 8 *)
WriteChar(CHR(12));      (* vspace = 12 *)
WriteChar('L');          (* Alpha cursor position identical with CAP *)
WriteChar(ALPHA);        (* exit in Alpha mode *)
END; (* ResetVDU *)

(******************************************************************************)

PROCEDURE InitVDU;

(* The dialogue region will be the top 4 text lines in Alpha mode.
   The window region will be the remaining screen area.
*)

BEGIN
DVIstatusl    := 1;              (* DVItoVDU assumes top text line = 1 *)
windowstatusl := 2;
messagel      := 3;
commandl      := 4;
maxy    := screenht - 1;
bottoml := screenht DIV lineht;  (* number of text lines on AED512 screen *)
windowv := 4 * lineht + 5;       (* height in pixels of dialogue region *)
windowh := 0;
windowht := screenht - windowv;
windowwd := 512;

WriteChar(INTER);
WriteChar(SBC);    (* make sure black is the background colour *)
WriteChar(black);
WriteChar(SEC);    (* change writing colour to green; easier to read than red *)
WriteChar(green);
WriteChar('(');    (* programmable option *)
WriteChar(CHR(6b));
WriteChar(CHR(1b));         (* override any overlapping TEK4010 commands,
                               regardless of option switch 4! *)
charwidth := charwidth1;    (* font for dialogue region *)
loadedwidth := charwidth;   (* for first StartGraphics call *)
SelectAlphaFont;
END; (* InitVDU *)
