/* -*-C-*- */
#include "texsgi.h"
#include "externs.h"
#include "proto.h"
char *getenv(char *s);

void
  dvifile(argc, argv, filestr)
int             argc;     /* argument count */
char           *argv[];   /* argument strings */
char           *filestr;  /* DVI filename to process */
{
  
#define PAGENUMBER(p) ((p) < 0 ? (page_count + 1 + (p)) : (p))
  
  register INT16  i, j, m;/* loop indices */
  register INT16  swap;   /* swap storage */
  
  INT16           m_begin, m_end, m_step;  /* loop limits and increment */
  INT16           list[3];/* index list for sort */
  char            tempstr[MAXSTR];  /* temporary string storage */
  
  
  /*
   * Establish the default  font file  search order.  This  is done
   * here instead of  in  initglob()  or  option(),  because  both
   * could  set fontlist[].
   * 
   * By default, the search list contains  names for the PK and GF font
   * files;  that  order   corresponds  to  increasing  size   (and
   * therefore, presumably, increasing  processing cost)  of font
   * raster information.  This search  order may  be overridden at  run
   * time  by defining an alternate one in the environment variable
   * FONTLIST;  the order of  the  strings  "PK" and  "GF"  in  that
   * variable determines the search  order, and  letter case  is NOT
   * significant. The  substrings  may  be  separated   by  zero  or
   * more   arbitrary characters, so  the  values  "PK-GF", "PK  GF",
   * "PKGF", "PK;GF", "PK/GF"  are all  acceptable,  and all  choose
   * the default search  order.   This  flexibility  in  separator
   * character choice is occasioned  by the  requirement on some
   * systems that  the environment variable have the syntax of a file
   * name, or be a  single "word".  If  any  substring  is  omitted,
   * then  that  font  will  be excluded from consideration.  Thus,
   * "GF" permits the use only of  GF font files.
   * 
   * The indexes gf_index, pk_index are in -1 .. 2, and at least one must
   * be non-negative.  A negative index excludes that font file type
   * from the search.
   */
  
  /* Note that non-negative entries in list[] are UNIQUE. */
  list[0] = gf_index = (INT16) strid2(fontlist, "GF");
  list[1] = pk_index = (INT16) strid2(fontlist, "PK");
  
  for (i = 0; i < 2; ++i)  /* put list in non-decreasing order */
    for (j = i + 1; j < 2; ++j)
      if (list[i] > list[j]) {
	swap = list[i];
	list[i] = list[j];
	list[j] = swap;
      }
  for (i = 0; i < 2; ++i)  /* assign indexes 0,1 */
    if (list[i] >= 0) {
      if (list[i] == gf_index)
	gf_index = i;
      else if (list[i] == pk_index)
	pk_index = i;
    }
  if ((gf_index < 0) && (pk_index < 0))
    (void) fatal("dvifile():  FONTLIST does not define at least one of \
GF or PK fonts");
  
  (void) dviinit(filestr);/* initialize DVI file processing */
  
  (void) devinit(argc, argv);  /* initialize device output */
  
  (void) readpost();
  (void) fseek(dvifp, 14L, 0);  /* position past preamble */
  (void) getbytes(dvifp, tempstr,
		  (BYTE) nosignex(dvifp, (BYTE) 1));  /* flush DVI comment */
  
  cur_page_number = 0;
  
  /* print pages in forward order for most devices */
  if (backwards) {
    m_begin = page_count;
    m_end = 1;
    m_step = -1;
  } else {
    m_begin = 1;
    m_end = page_count;
    m_step = 1;
  }
  
  for (i = 0; i < npage; ++i) {
    /*
     * make page numbers positive and
     * order pairs non-decreasing
     */
    page_begin[i] = PAGENUMBER(page_begin[i]);
    page_end[i] = PAGENUMBER(page_end[i]);
    if (page_begin[i] > page_end[i]) {
      swap = page_begin[i];
      page_begin[i] = page_end[i];
      page_end[i] = swap;
    }
  }
  
  for (m = m_begin;; m += m_step) {
    for (i = 0; i < npage; ++i)  /* search page list */
      if (IN(page_begin[i], m, page_end[i]) &&
    (((m - page_begin[i]) % page_step[i]) == 0)) {
  if (!quiet)  /* start progress report */
    (void) fprintf(stderr, "[%d", m);
  cur_page_number++;  /* sequential page number 1..N */
  cur_index = m - 1;  /* remember index globally */
  prtpage(page_ptr[cur_index]);
  if (!quiet)  /* finish progress report */
    (void) fprintf(stderr, "] ");
  break;
      }
    if (m == m_end)  /* exit loop after last page */
      break;
  }
  
  (void) devterm();  /* terminate device output */
  
  (void) dviterm();  /* terminate DVI file processing */
}

void
  dviinit(filestr)    /* initialize DVI file processing */
char           *filestr;/* command line filename string  */
{
  int             i;  /* temporary */
  INT16           k;  /* index in curpath[] and curname[] */
  char           *tcp;  /* temporary string pointers */
  char           *tcp1;  /* temporary string pointers */
  
  /***********************************************************************
    Unix allows binary I/O on stdin and stdout; most other operating systems
    do not.  In order  to support this,  we use the  convention that a  null
    filestr (NOT  a (char*)NULL)  means the  dvifile is  on stdin,  and  the
    translated output is on stdout.  Error logging will be to dvixxx.err  or
    dvixxx.dvi-err.  Because random access is required of the dvifile, stdin
    may NOT be a pipe; it must be a disk file.
    ***********************************************************************/
#define NO_FILENAME (filestr[0] == '\0')
  
  for (k = 0; k < 10; ++k)
    tex_counter[k] = (INT32) 0L;
  
  fontptr = (struct font_entry *) NULL;
  hfontptr = (struct font_entry *) NULL;
  pfontptr = (struct font_entry *) NULL;
  fontfp = (FILE *) NULL;
  cache_size = 0;
  nopen = 0;
  
  /***********************************************************
    Set up file names and open dvi and plot files.  Simple
    filename parsing assumes forms like:
    
    Unix:  name
    name.ext
    /dir/subdir/.../subdir/name
    /dir/subdir/.../subdir/name.ext
    
    We only  need  to  extract the  directory  path,  name,  and
    extension, so the parsing is simple-minded.
    
    ***********************************************************/
  
  tcp = strrchr(filestr, '/');  /* find end of Unix file path */
  
  if (tcp == (char *) NULL) {  /* no file path */
    curpath[0] = '\0';  /* empty path */
    tcp = filestr;  /* point to start of name */
  } else {    /* save path for later use */
    k = (INT16) (tcp - filestr + 1);
    (void) strncpy(curpath, filestr, (int) k);
    curpath[k] = '\0';
    tcp++;    /* point to start of name */
  }
  
  tcp1 = strrchr(tcp, '.');  /* find last dot in filename */
  
  if (tcp1 == (char *) NULL) {  /* no dot, so name has no extension */
    (void) strcpy(curname, tcp);  /* save name */
    tcp1 = strchr(tcp, '\0');  /* set empty extension */
  } else {    /* save name part */
    k = (INT16) (tcp1 - tcp);
    (void) strncpy(curname, tcp, (int) k);
    curname[k] = '\0';
  }
  
  (void) strcpy(curext, tcp1);  /* save extension */
  
  
  /*
   * DVI file must always have extension DVIEXT; supply one if
   * necessary (e.g /u/jones/foo.dvi)
   */
  
  (void) strcpy(dviname, curpath);
  (void) strcat(dviname, curname);
  if (curext[0] == '\0')
    (void) strcat(dviname, DVIEXT);
  else
    (void) strcat(dviname, curext);
  
  /*
   * Font substitution file is PATH NAME SUBEXT (e.g.
   * /u/jones/foo.sub).  We do not tell user (via stderr) about it
   * until it is needed.
   */
  
  if (subfile[0] == '\0') {  /* no -fsubfile; make default */
    (void) strcpy(subfile, curpath);
    (void) strcat(subfile, curname);
    (void) strcat(subfile, subext);
  }
  if (NO_FILENAME) {
    dvifp = stdin;  /* already open, so no DEBUG_OPEN call */
    if (fseek(dvifp, 0L, 0)) {
      (void) fprintf(stderr,
         "%s: Unable to fseek() on stdin--was it a pipe?\n",
         g_progname);
      (void) exit(1);
    }
  } else {
    dvifp = fopen(dviname, "r");
    DEBUG_OPEN(dvifp, dviname, "r");
  }
  
  if (dvifp == (FILE *) NULL) {
    (void) sprintf(message, "dviinit(): %s: can't open [%s]",
       g_progname, dviname);
    (void) fatal(message);
  }
  strcpy(dvoname, "stdout");  /* BitGraph always goes to stdout */
  plotfp = stdout;
  
  if (!quiet && !NO_FILENAME) {
    (void) fprintf(stderr, "[Input from DVI file %s]\n", dviname);
  }
  g_logname[0] = '\0';  /* no PATH on output file */
  
  if (NO_FILENAME) {  /* use texsgi as file prefix */
    (void) strcpy(g_logname, "texsgi");
  }
  (void) strcat(g_logname, curname);
  (void) strcat(g_logname, ".");
  (void) strcat(g_logname, DVIPREFIX);
  
  (void) strcat(g_logname, "err");
  
  lmargin = (COORDINATE) (leftmargin * ((float) XDPI));
  tmargin = (COORDINATE) (topmargin * ((float) YDPI));
  
  if ((BYTE) nosignex(dvifp, (BYTE) 1) != PRE)
    (void) fatal("dviinit(): PRE doesn't occur first--are you sure \
this is a DVI file?");
  
  i = (int) signex(dvifp, (BYTE) 1);
  if (i != DVIFORMAT) {
    (void) sprintf(message,
       "dviinit(): DVI format = %d, can only process DVI format %d files.",
       i, (int) DVIFORMAT);
    (void) fatal(message);
  }
}

void
  dviterm()
{        /* terminate DVI file processing */
  register INT16  k;  /* loop index */
  
  if (fflush(plotfp) == EOF)
    (void) fatal("dviterm(): Output error -- disk storage probably full");
  
  /*
   * Close all files and free dynamically-allocated font space.  Stdin
   * is never closed, however, because it might be needed later for
   * ioctl().
   */
  
  if ((dvifp != (FILE *) NULL) && (dvifp != stdin))
    (void) fclose(dvifp);
  if (plotfp != (FILE *) NULL)
    (void) fclose(plotfp);
  
  for (k = 1; k <= (INT16) nopen; ++k) {
    if (font_files[k].font_id != (FILE *) NULL) {
      
#if    VIRTUAL_FONTS
      if (virt_font)
  (void) virtfree(font_files[k].font_id);
#endif        /* VIRTUAL_FONTS */
      
      (void) fclose(font_files[k].font_id);
      font_files[k].font_id = (FILE *) NULL;
    }
  }
  
  fontptr = hfontptr;
  while (fontptr != (struct font_entry *) NULL) {
    for (k = 0; k < NPXLCHARS; ++k)
      if ((fontptr->ch[k].rasters) != (UNSIGN32 *) NULL)
  (void) free((char *) fontptr->ch[k].rasters);
    pfontptr = fontptr;
    fontptr = fontptr->next;
    (void) free((char *) pfontptr);
  }
  if (!quiet) {
    (void) fprintf(stderr, " [OK]\n");
  }
#if    BSD42
  if (spool_output) {
    register char  *p;
    
    (void) sprintf(message, "DVISPOOL %s\n", dvoname);
    
    for (p = message; *p; ++p) {
      (void) ioctl(fileno(stdin), TIOCSTI, p);
      
    }
  }
#endif        /* BSD42 */
  
}

void
  findpost()
{
  register long   postambleptr;
  register BYTE   i;
  
  (void) fseek(dvifp, 0L, 2);  /* goto end of file */
  postambleptr = ftell(dvifp) - 4;
  (void) fseek(dvifp, postambleptr, 0);
  
  while (TRUE) {
    (void) fseek(dvifp, --(postambleptr), 0);
    if (((i = (BYTE) nosignex(dvifp, (BYTE) 1)) != 223) &&
  (i != DVIFORMAT))
      (void) fatal("findpost():  Bad end of DVI file");
    if (i == DVIFORMAT)
      break;
  }
  (void) fseek(dvifp, postambleptr - 4, 0);
  (void) fseek(dvifp, (long) nosignex(dvifp, (BYTE) 4), 0);
}

COORDINATE
  fixpos(cc, c, cnvfac)
register COORDINATE cc;  /* coordinates in device pixels */
register INT32  c;  /* coordinates in DVI units */
register float  cnvfac;  /* converts DVI units to pixels */
{
  register COORDINATE ccc;
  
  /*
   * A sequence  of consecutive  rules, or  consecutive characters  in
   * a fixed-width font whose width is not an integer number of pixels,
   * can cause |cc| to  drift far away  from a correctly  rounded
   * value.   We follow DVITYPE Version 2.6 to ensure  that the amount
   * of drift  will never exceed MAXDRIFT pixels.  DVITYPE Version 2.0
   * did not do  this, and the results could be visibly AWFUL!!
   */
  
  ccc = PIXROUND(c, cnvfac);
  if (ABS(ccc - cc) > MAXDRIFT) {  /* drag cc toward the correctly
           * rounded value ccc */
    if (ccc > cc) {
      cc = ccc - MAXDRIFT;
    } else {
      cc = ccc + MAXDRIFT;
    }
  }
  return (cc);
}

void
  movedown(a)
register INT32  a;

{
  /*
   * From DVITYPE Version 2.6: "Vertical motion is done similarly [to
   * horizontal motion handled in moveover()],  but with  the threshold
   * between ``small''  and ``large'' increased by  a factor of  five.
   * The idea  is to  make fractions  like  ``1/2''  round
   * consistently,  but  to   absorb accumulated rounding errors in the
   * baseline-skip moves."
   * 
   * The one precaution we need to  take here is that fontptr can  be
   * NULL, which we treat like  a large movement.  This NULL  pointer
   * was used without error  on many different  machines for 2  years
   * before it was caught on the VAX VMS implementation, which  makes
   * memory page 0 inaccessible.
   */
  v += a;
  if ((fontptr == (struct font_entry *) NULL) ||
      (ABS(a) >= 5 * fontptr->font_space))
    vv = PIXROUND(v, conv) + tmargin;
  else {
    vv += PIXROUND(a, conv);
    vv = fixpos(vv - tmargin, v, conv) + tmargin;
  }
}

void
  moveover(b)
register INT32  b;

{
  /*
   * From DVITYPE Version 2.6: "Rounding to the nearest pixel is  best
   * done in the manner shown here, so as to  be inoffensive to the
   * eye: When the  horizontal motion is small, like a kern, |hh|
   * changes by rounding the kern; but when the motion is large, |hh|
   * changes by rounding the  true position |h| so that  accumulated
   * rounding errors disappear.  We allow a  larger space  in  the
   * negative  direction than  in  the positive one, because TeX  makes
   * comparatively large  backspaces when it positions accents."
   * 
   * The one precaution we need to  take here is that fontptr can  be
   * NULL, which we treat like  a large movement.  This NULL  pointer
   * was used without error  on many different  machines for 2  years
   * before it was caught on the VAX VMS implementation, which  makes
   * memory page 0 inaccessible.
   */
  h += b;
  if ((fontptr == (struct font_entry *) NULL) ||
      ABS(b) >= fontptr->font_space)
    hh = PIXROUND(h, conv) + lmargin;
  else {
    hh += PIXROUND(b, conv);
    hh = fixpos(hh - lmargin, h, conv) + lmargin;
  }
}

void
  moveto(x, y)
COORDINATE      x, y;

{
  xcp = x;
  ycp = y;
}

void
  newfont()
{
  register UNSIGN16 the_char;  /* loop index */
  register struct char_entry *tcharptr;
  
  fullfont = (fullfont + 1) % NBGFONTS;
  if (fullfont == 0) {
    partfont = NBGFONTS;
    partchar = FIRSTBGCHAR;
  } else {
    partchar += FIRSTBGCHAR;
    if (partchar > LASTBGCHAR) {
      partfont++;
      partchar = FIRSTBGCHAR;
    }
  }
  for (the_char = FIRSTPXLCHAR; the_char <= LASTPXLCHAR; the_char++) {
    tcharptr = &(fontptr->ch[the_char]);
    tcharptr->isloaded = FALSE;
    if (the_char < FIRSTBGCHAR) {
      tcharptr->bgfont = partfont;
      tcharptr->bgchar = partchar + the_char;
    } else {
      tcharptr->bgfont = fullfont;
      tcharptr->bgchar = the_char;
    }
  }
}

UNSIGN32
  nosignex(fp, n)      /* return n byte quantity from file fd */
register FILE  *fp;  /* file pointer    */
register BYTE   n;  /* number of bytes (1..4) */
     
{
  register UNSIGN32 number;  /* number being constructed */
  
  number = 0;
  while (n--) {
    number <<= 8;
    number |= getc(fp);
  }
  return ((UNSIGN32) number);
}

void
  openfont(fontname)
char           *fontname;

/***********************************************************************
  The original version of this DVI driver reopened the font file  each
  time the font changed, resulting in an enormous number of relatively
  expensive file  openings.  This version  keeps  a cache  of  up  to
  MAXOPEN open files,  so that when  a font change  is made, the  file
  pointer, fontfp, can  usually be  updated from the  cache.  When the
  file is not found in  the cache, it must  be opened.  In this  case,
  the next empty slot  in the cache  is assigned, or  if the cache  is
  full, the least used font file is closed and its slot reassigned for
  the new file.  Identification of the least used file is based on the
  counts of the number  of times each file  has been "opened" by  this
  routine.  On return, the file pointer is always repositioned to  the
  beginning of the file.
  
  If the first open attempt  fails, an attempt will  be made to use  a
  substitute font, then neighboring magnifications (with the same font
  name), or substitutes for them.
  ***********************************************************************/

{
  register INT16  i, j, k;/* loop indices */
  INT16           current;
  INT16           least_used;
  INT16           maxopen = MAXOPEN;
  
  struct font_entry *tfontptr;
  char            subfont[MAXFNAME];
  int             submag;
  char           *filelist[MAXFORMATS];  /* pointers to templist[][] */
  char            templist[MAXFORMATS][MAXFNAME];
  
#if    VIRTUAL_FONTS
  struct stat     statbuf;/* so fstat() can get file size */
  char           *p;
#endif
  
  if ((pfontptr != (struct font_entry *) NULL) && (pfontptr == fontptr))
    return;    /* we need not have been called */
  
  for (j = 0; j < MAXFORMATS; ++j)  /* initialize fontlist
           * pointers */
    filelist[j] = &templist[j][0];
  
  for (current = 1;
       (current <= nopen) &&
       (font_files[current].font_id != fontptr->font_file_id);
       ++current);  /* try to find file in open list */
  
  if (current <= nopen)  /* file already open, lookup its id */
    fontfp = font_files[current].font_id;
  else {
    /***************************************************************
      The file was  not in list  of open  files.  If the  list is  not
      full, add it to  the list; otherwise  close the least-used  file
      and remove it from the font_entry containing it.  Finally,  open
      the file, or its closest  neighbor in the magnification  family.
      A warning is issued  if no file can  be opened.  The caller  can
      then proceed with zero font metrics if desired.
      ***************************************************************/
    
    if (nopen < maxopen)  /* just add it to list */
      current = ++nopen;
    else {    
      /* list full -- find least used file,
       * close it, and reuse slot for new file
       */
      least_used = 1;
      for (i = 2; i <= maxopen; ++i)
	if (font_files[least_used].use_count >
	    font_files[i].use_count)
	  least_used = i;
      
      fontfp = font_files[least_used].font_id;
      tfontptr = hfontptr;
      while (tfontptr != (struct font_entry *) NULL) { 
	/*
	 * remove file pointer
	 * from its font_entry
	 */
	if (tfontptr->font_file_id == fontfp) {
	  tfontptr->font_file_id = (FILE *) NULL;
	  break;
	}
	tfontptr = tfontptr->next;
      }
      
#if    VIRTUAL_FONTS
      if (virt_font && (fontfp != (FILE *) NULL))
	(void) virtfree(fontfp);
#endif
      
      (void) fclose(fontfp);
      fontfp = (FILE *) NULL;
      current = least_used;
    }
    (void) actfact(fontptr->font_mag);  /* Get global mag_index */
#ifdef DEBUG1
    fprintf(stderr,"requesting: %s at %f; actfact: %f, MAGSIZE: %f\n",
	    fontname, fontptr->font_mag,
	    actfact(fontptr->font_mag),
	    (int)MAGSIZE(mag_table[mag_index + i]));
#endif
    
    fontfp = (FILE *) NULL;
    
    /***************************************************************
      Try the requested font, then any substitute font, then for  each
      neighboring magnification  from  nearest to  furthest,  try  the
      requested font, and then any substitute font.
      ***************************************************************/
    
    for (k = 0; (fontfp == (FILE *) NULL) && (k < magtablesize); ++k) {
      /* loop over mag family */
      for (i = -k; (fontfp == (FILE *) NULL) && (i <= k); i += MAX(1, k + k)) {
	/* try smaller, then larger */
	if (IN(0, mag_index + i, magtablesize - 1)) {
#ifdef DEBUG1
	  fprintf(stderr,"magtable[%d] = %f, MAGSIZE: %d\n",
		  (mag_index + i), mag_table[mag_index + i], 
		  (int)MAGSIZE(mag_table[mag_index + i]));
#endif
	  (void) fontfile(filelist, ((fontptr->a == 0) ? fontpath : ""),
			  fontname, (int) MAGSIZE(mag_table[mag_index + i]));
	  for (j = 0; (j < MAXFORMATS) && *filelist[j]; ++j) {
	    fontfp = fopen(filelist[j], "r");
	    DEBUG_OPEN(fontfp, filelist[j], "r");
	    if (fontfp != (FILE *) NULL) {
	      strcpy(fontptr->name, filelist[j]);
	      break;
	    }
	  }
	  if ((k > 0) && (fontfp != (FILE *) NULL)) {
	    (void) sprintf(message,
			   "Font [%s [mag %d]] substituted with [%s [mag %d]].",
			   fontname, (int) MAGSIZE(mag_table[mag_index]),
			   fontname, (int) MAGSIZE(mag_table[mag_index + i]));
	    (void) warning(message);
	  }
	  if ((fontfp == (FILE *) NULL) && 
	      fontsub(subfont, &submag, fontname,
		      (int) MAGSIZE(mag_table[mag_index + i]))) {
	    (void) fontfile(filelist, ((fontptr->a == 0) ? fontpath : ""),
			    subfont, (submag ? submag :
				      (int) MAGSIZE(mag_table[mag_index + i])));
	    for (j = 0; (j < MAXFORMATS) && *filelist[j]; ++j) {
	      fontfp = fopen(filelist[j], "r");
	      DEBUG_OPEN(fontfp, filelist[j], "r");
	      if (fontfp != (FILE *) NULL) {
		strcpy(fontptr->name, filelist[j]);
		break;
	      }
	    }
	    
	    if (fontfp != (FILE *) NULL) {
	      (void) sprintf(message,
			     "Substituting font file [%s [mag %d]] \
by [%s [mag %d]]",
			     fontname, (int) MAGSIZE(mag_table[mag_index]),
			     fontptr->name,
			     (int) MAGSIZE(mag_table[mag_index + i]));
	      (void) warning(message);
	    }
	  }
	}
      }  /* end for (i) -- loop over smaller and larger neighbors  */
    }    /* end for (k) -- loop over mag family */
    
    if (fontfp == (FILE *) NULL) {
      --nopen;/* don't count this failed open */
      (void) sprintf(message, "Font file [%s [mag %d]] could not be \
opened; %d font files are open\n\
Proceeding with zero size characters for this font",
         fontname, (int) MAGSIZE(mag_table[mag_index]), nopen);
      (void) warning(message);
    }
    font_files[current].font_id = fontfp;
    font_files[current].use_count = 0;
    
#if    VIRTUAL_FONTS
    /*
     * ***************************************************************
     * 
     * This code  is implementation-dependent.   On many  C
     * compilers, FILE points to a struct of the form
     * 
     * struct  _iobuf { char  *_ptr;  // pointer to next available
     * char int  _cnt;  // number of chars left in buffer
     * char  *_base;  // pointer to start of buffer int
     * _flag;  // assorted flags int  _file;  // file
     * number handle }
     * 
     * To implement virtual fonts,  we save the  pointers in a
     * private global array, get the file size from fstat(),
     * malloc() a  buffer for it, read the entire file in, and
     * replace the pointers in the FILE variable with ones  for
     * the  new buffer.  Just before  the file is  closed, the
     * space is  freed  and the   old pointers are restored.
     * Although many C implementations use  malloc() to get the
     * buffer  space  in the  first place, which  would  make
     * this saving unnecessary, not all do; see  the
     * implementation given in Kernighan and Ritchie "The C
     * Programming Language", p. 168.
     * 
     * In  implementing   this   code   on   any   new   machine,
     * the implementations of fread(), read(), fseek(), and
     * rewind() should be checked to  choose the  most efficient
     * one  (e.g. one  system call for entire file  read, and no
     * double buffering), and  make sure that  fseek() and
     * rewind()  do not  unnecessarily  discard input buffer
     * contents.  fseek()  and  rewind() are  defined  in terms
     * of macros fseek() and rewind() in machdefs.h to facilitate
     * replacement.
     ***************************************************************
     * 
     */
    if (virt_font) {
      virt_save[fileno(fontfp)].base = (char *) NULL;
      (void) fstat(fileno(fontfp), &statbuf);  /* get file size */
      p = (char *) malloc((int) statbuf.st_size);  /* get file buffer */
      if (p != (char *) NULL) {
  if (read(fileno(fontfp), p, (int) statbuf.st_size)
      == (int) statbuf.st_size) {  /* read successful */
    virt_save[fileno(fontfp)].ptr = FILE_PTR(fontfp);
    virt_save[fileno(fontfp)].cnt = FILE_CNT(fontfp);
    virt_save[fileno(fontfp)].base = FILE_BASE(fontfp);
    FILE_PTR(fontfp) = p;
    FILE_BASE(fontfp) = p;
    FILE_CNT(fontfp) = (int) statbuf.st_size;
  } else {  /* failure */
    (void) rewind(fontfp);
    (void) free(p);  /* free dynamic buffer */
  }
      }
      if (DBGOPT(DBG_FONT_CACHE)) {
  (void) fprintf(stderr, "\nopenfont(): Font file %d [%s] ",
           (int) fileno(fontfp), fontptr->name);
  if (p == (char *) NULL)
    (void) fprintf(stderr,
       "cannot callocate buffer for entire file\n");
  else
    (void) fprintf(stderr,
       "buffer length 0x%x\told buffer at 0x%lx\t\
new buffer at 0x%lx\n",
       (int) statbuf.st_size,
       (long) virt_save[fileno(fontfp)].base,
       (long) FILE_BASE(fontfp));
      }
    }
#endif
    
  }      /* end if (file is in open list) */
  
  pfontptr = fontptr;  /* make previous = current font */
  fontptr->font_file_id = fontfp;  /* set file identifier */
  font_files[current].use_count++;  /* update reference count */
}

#if    VIRTUAL_FONTS
void
  virtfree(fp)      /* free buffer space before close */
FILE           *fp;
{
  if (virt_font && (fp != (FILE *) NULL) &&
      (virt_save[fileno(fp)].base != (char *) NULL)) {
    (void) fflush(fp);
    (void) free(FILE_BASE(fp));
    FILE_PTR(fp) = virt_save[fileno(fp)].ptr;
    FILE_CNT(fp) = virt_save[fileno(fp)].cnt;
    FILE_BASE(fp) = virt_save[fileno(fp)].base;
    virt_save[fileno(fp)].base = (char *) NULL;
  }
}
#endif

void
  prtpage(bytepos)    /* print page whose BOP is at bytepos */
long            bytepos;

{
  struct stack_entry {
    INT32           h;
    COORDINATE      hh;
    INT32           v;
    COORDINATE      vv;
    INT32           w, x, y, z;  /* what's on stack */
  };
  register BYTE   command;/* current command         */
  register INT16  i;  /* command parameter; loop index     */
  char            tc;  /* temporary character       */
  UNSIGN32        ht_rule;/* rule height                              */
  UNSIGN32        wd_rule;/* rule width                               */
  INT32           k, m;  /* temporary parameter       */
  BOOLEAN         seen_bop;  /* flag for noting processing of BOP   */
  register INT16  sp;  /* stack pointer         */
  struct stack_entry stack[STACKSIZE];  /* stack       */
  char            specstr[MAXSPECIAL + 1];  /* \special{} string     */
  INT32           w;  /* current horizontal spacing     */
  INT32           x;  /* current horizontal spacing     */
  INT32           y;  /* current vertical spacing       */
  INT32           z;  /* current vertical spacing       */
  
  /***********************************************************************
    Process all commands  between the  BOP at bytepos  and the  next BOP  or
    POST.  The page is  printed when the  EOP is met,  but font changes  can
    also happen between EOP and BOP, so we have to keep going after EOP.
    ***********************************************************************/
  
  seen_bop = FALSE;  /* this is first time through */
  (void) fseek(dvifp, bytepos, 0);  /* start at the desired
           * position */
  
  for (;;) {
    /* "infinite" loop - exits when POST or second BOP met */
    int             dev;
    
    if ((dev = qtest()) != 0 && dev != INPUTCHANGE)  /* some queued event */
      command = EOP;  /* so set for end-of-page action */
    else
      command = (BYTE) nosignex(dvifp, (BYTE) 1);
    switch (command) {
      
    case SET1:
    case SET2:
    case SET3:
    case SET4:
      (void) setchar((BYTE) nosignex(dvifp, (BYTE) (command - SET1 + 1)), TRUE);
      break;
      
    case SET_RULE:
      ht_rule = nosignex(dvifp, (BYTE) 4);
      wd_rule = nosignex(dvifp, (BYTE) 4);
      (void) setrule(ht_rule, wd_rule, TRUE);
      break;
      
    case PUT1:
    case PUT2:
    case PUT3:
    case PUT4:
      (void) setchar((BYTE) nosignex(dvifp, (BYTE) (command - PUT1 + 1)), FALSE);
      break;
      
    case PUT_RULE:
      ht_rule = nosignex(dvifp, (BYTE) 4);
      wd_rule = nosignex(dvifp, (BYTE) 4);
      (void) setrule(ht_rule, wd_rule, FALSE);
      break;
      
    case NOP:
      break;
      
    case BOP:
      if (seen_bop)
	return;  /* we have been here already */
      seen_bop = TRUE;
      
      for (i = 0; i <= 9; i++)
	tex_counter[i] = (INT32) signex(dvifp, (BYTE) 4);
      /*
       *display menu at top of
       * page
       */
      (void) bopact();

      h = v = w = x = y = z = 0;
      hh = lmargin;
      vv = tmargin;
      sp = 0;
      fontptr = (struct font_entry *) NULL;
      break;
      
    case EOP:
      
      (void) eopact();
      return;
      
    case PUSH:
      if (sp >= STACKSIZE)
  (void) fatal("prtpage():  stack overflow");
      stack[sp].h = h;
      stack[sp].hh = hh;
      stack[sp].v = v;
      stack[sp].vv = vv;
      stack[sp].w = w;
      stack[sp].x = x;
      stack[sp].y = y;
      stack[sp].z = z;
      sp++;
      break;
      
    case POP:
      --sp;
      if (sp < 0)
	(void) fatal("prtpage():  stack underflow");
      h = stack[sp].h;
      hh = stack[sp].hh;
      v = stack[sp].v;
      vv = stack[sp].vv;
      w = stack[sp].w;
      x = stack[sp].x;
      y = stack[sp].y;
      z = stack[sp].z;
      break;
      
    case RIGHT1:
    case RIGHT2:
    case RIGHT3:
    case RIGHT4:
      (void) moveover(signex(dvifp, (BYTE) (command - RIGHT1 + 1)));
      break;
      
    case W0:
      (void) moveover(w);
      break;
      
    case W1:
    case W2:
    case W3:
    case W4:
      w = (INT32) signex(dvifp, (BYTE) (command - W1 + 1));
      (void) moveover(w);
      break;
      
    case X0:
      (void) moveover(x);
      break;
      
    case X1:
    case X2:
    case X3:
    case X4:
      x = (INT32) signex(dvifp, (BYTE) (command - X1 + 1));
      (void) moveover(x);
      break;
      
    case DOWN1:
    case DOWN2:
    case DOWN3:
    case DOWN4:
      (void) movedown(signex(dvifp, (BYTE) (command - DOWN1 + 1)));
      break;
      
    case Y0:
      (void) movedown(y);
      break;
      
    case Y1:
    case Y2:
    case Y3:
    case Y4:
      y = signex(dvifp, (BYTE) (command - Y1 + 1));
      (void) movedown(y);
      break;
      
    case Z0:
      (void) movedown(z);
      break;
      
    case Z1:
    case Z2:
    case Z3:
    case Z4:
      z = signex(dvifp, (BYTE) (command - Z1 + 1));
      (void) movedown(z);
      break;
      
    case FNT1:
    case FNT2:
    case FNT3:
    case FNT4:
      (void) setfntnm((INT32) nosignex(dvifp,
               (BYTE) (command - FNT1 + 1)));
      break;
      
    case XXX1:
    case XXX2:
    case XXX3:
    case XXX4:
      k = (INT32) nosignex(dvifp, (BYTE) (command - XXX1 + 1));
      if (k > MAXSPECIAL) {
  (void) sprintf(message,
           "prtpage():  \\special{} string of %d characters longer \
than DVI driver limit of %d -- truncated.",
           k, MAXSPECIAL);
  (void) warning(message);
      }
      m = 0;
      while (k--) {
  tc = (char) nosignex(dvifp, (BYTE) 1);
  if (m < MAXSPECIAL)
    specstr[m++] = tc;
      }
      specstr[m] = '\0';
      (void) special(specstr);
      break;
      
    case FNT_DEF1:
    case FNT_DEF2:
    case FNT_DEF3:
    case FNT_DEF4:
      if (preload)
	(void) skipfont((INT32) nosignex(dvifp,
					 (BYTE) (command - FNT_DEF1 + 1)));
      else
	(void) readfont((INT32) nosignex(dvifp,
					 (BYTE) (command - FNT_DEF1 + 1)));
      break;
      
    case PRE:
      (void) fatal("prtpage():  PRE occurs within file");
      break;
      
    case POST:
      (void) devterm();  /* terminate device output */
      (void) dviterm();  /* terminate DVI file
       * processing */
      (void) alldone();  /* this will never return */
      break;
      
    case POST_POST:
      (void) fatal("prtpage():  POST_POST with no preceding POST");
      break;
      
    default:
      if (command >= FONT_00 && command <= FONT_63)
  (void) setfntnm((INT32) (command - FONT_00));
      else if (command >= SETC_000 && command <= SETC_127)
  (void) setchar((BYTE) (command - SETC_000), TRUE);
      else {
  (void) sprintf(message, "prtpage():  %d is an undefined command",
           command);
  (void) fatal(message);
      }
      break;
    }
  }
}

void
  readfont(font_k)
INT32           font_k;
{
  BYTE            a, l;
  UNSIGN32        c;  /* checksum */
  UNSIGN32        d;  /* design size */
  char            n[MAXSTR];
  UNSIGN32        s;  /* scale factor */
  struct font_entry *tfontptr;  /* temporary font_entry pointer */
  
  c = nosignex(dvifp, (BYTE) 4);
  s = nosignex(dvifp, (BYTE) 4);
  d = nosignex(dvifp, (BYTE) 4);
  a = (BYTE) nosignex(dvifp, (BYTE) 1);
  l = (BYTE) nosignex(dvifp, (BYTE) 1);
  (void) getbytes(dvifp, n, (BYTE) (a + l));
  n[a + l] = '\0';
  tfontptr = (struct font_entry *) malloc((unsigned) sizeof(struct font_entry));
  if (tfontptr == (struct font_entry *) NULL)
    (void) fatal(
     "readfont():  No allocable memory space left for font_entry");
  tfontptr->next = hfontptr;
  
  fontptr = hfontptr = tfontptr;
  fontptr->k = font_k;
  fontptr->c = c;
  fontptr->s = s;
  fontptr->d = d;
  fontptr->a = a;
  fontptr->l = l;
  (void) strcpy(fontptr->n, n);
  fontptr->font_space = (INT32) (s / 6);
  (void) reldfont(fontptr);
}

void
  readpost()
/***********************************************************************
  This routine is used  to read in the  postamble values.  It  initializes
  the magnification and checks the stack height prior to starting printing
  the document.
  ***********************************************************************/

{
  long            lastpageptr;  /* byte pointer to last physical page */
  int             the_page_count;  /* page count from DVI file */
  
  findpost();
  if ((BYTE) nosignex(dvifp, (BYTE) 1) != POST)
    (void) fatal("readpost():  POST missing at head of postamble");
  
  lastpageptr = (long) nosignex(dvifp, (BYTE) 4);
  num = nosignex(dvifp, (BYTE) 4);
  den = nosignex(dvifp, (BYTE) 4);
  mag = nosignex(dvifp, (BYTE) 4);
#ifdef DEBUG1
  fprintf(stderr,"postamble: num: %d, den: %d, mag: %d\n", (int)num,
	  (int)den, (int)mag);
#endif
  conv = ((float) num / (float) den) *
    ((float) runmag / (float) STDMAG) *
      
#if    USEGLOBALMAG
      actfact(mag) *
#endif
  
	((float) RESOLUTION / 254000.0);
#ifdef DEBUG1
  fprintf(stderr, "conv: %f\n", (float)conv);
#endif
  
  /*
   * denominator/numerator here since will be dividing by the
   * conversion factor
   */
  (void) nosignex(dvifp, (BYTE) 4);  /* height-plus-depth of
           * tallest page */
  (void) nosignex(dvifp, (BYTE) 4);  /* width of widest page */
  if ((int) nosignex(dvifp, (BYTE) 2) >= STACKSIZE)
    (void) fatal("readpost():  Stack size is too small");
  
  the_page_count = (int) nosignex(dvifp, (BYTE) 2);
  
  if (!quiet) {
    (void) fprintf(stderr, "[%d pages]\n", the_page_count);
    (void) fprintf(stderr, "[%d magnification]\n", (int) runmag);
  }
  if (preload)
    getfntdf();
  getpgtab(lastpageptr);
}

COORDINATE
  rulepxl(number, cnvfac)    /* return number of pixels in a rule */
register UNSIGN32 number;  /* in DVI units     */
register float  cnvfac;  /* conversion factor */

{
  register COORDINATE n;
  
  n = (COORDINATE) (number * cnvfac);
  if ((float) n < ((float) (number)) * cnvfac)
    return ((COORDINATE) (n + 1));
  else
    return ((COORDINATE) n);
}

void
  setrule(height, width, update_h)
register UNSIGN32 height, width;
register BOOLEAN update_h;

{        /* draw a rule with bottom left corner at
         * (h,v) */
  if ((height > 0) && (width > 0))  /* non-empty rule */
    fillrect(hh + xscreen, win_y_size - vv + yscreen,
       rulepxl(width, conv), rulepxl(height, conv));
  
  if (update_h) {
    h += (INT32) width;
    hh += rulepxl(width, conv);
    hh = fixpos(hh - lmargin, h, conv) + lmargin;
  }
}

INT32
  signex(fp, n)      /* return n byte quantity from file fd */
register FILE  *fp;  /* file pointer   */
register BYTE   n;  /* number of bytes */

{
  register BYTE   n1;  /* number of bytes    */
  register INT32  number;  /* number being constructed */
  
  number = (INT32) getc(fp);  /* get first (high-order) byte */
  
#if    OS_UNIX      /* 4.1BSD C compiler uses logical shift too */
  if (number > 127)  /* sign bit is set, pad top of word with sign
       * bit */
    number |= 0xffffff00;
#endif        /* OS_UNIX */
  
  n1 = n--;
  while (n--) {
    number <<= 8;
    number |= (INT32) getc(fp);
  }
  
  return ((INT32) number);
}

void
  skipfont(k)
INT32           k;  /* UNUSED */

{
  BYTE            a, l;
  char            n[MAXSTR];
  
  (void) nosignex(dvifp, (BYTE) 4);
  (void) nosignex(dvifp, (BYTE) 4);
  (void) nosignex(dvifp, (BYTE) 4);
  a = (BYTE) nosignex(dvifp, (BYTE) 1);
  l = (BYTE) nosignex(dvifp, (BYTE) 1);
  getbytes(dvifp, n, (BYTE) (a + l));
}

char           *
  tctos()
{
  /* 
   * return pointer to (static) TeX page counter string 
   * (trailing zero counters are not printed) 
   */
  register INT16  k;  /* loop index */
  register INT16  n;  /* number of counters to print */
  static char     s[111];  /* 10 32-bit counters n.n.n... */
  
  for (n = 9; (n > 0) && (tex_counter[n] == 0); --n)
    /* NO-OP */ ;
  s[0] = '\0';
  for (k = 0; k <= n; ++k)
    (void) sprintf(strrchr(s, '\0'), "%ld%s",
       tex_counter[k], (k < n) ? "." : "");
  return ((char *) &s[0]);
}

