# browser.tcl -- Make TkDVI browser windows.
# Copyright  1999 Anselm Lingnau <lingnau@tm.informatik.uni-frankfurt.de>.
# See file COPYING for conditions on use and distribution.
# $Id: browser.tcl,v 1.1.1.1 1999/06/10 12:36:50 lingnau Exp $

package provide tkdvi::browser 0.1

namespace eval tkdvi::browser {
    proc open {top} {
	if {[string compare $top {}] == 0} {
	    set count 0
	    set top ".tkdvi[incr count]"
	    while {[winfo exists $top]} {
		set top ".tkdvi[incr count]"
	    }
	}
	set types {
	    {{DVI Files} {.dvi}}
	    {{All Files} *}
	}
	set name [tk_getOpenFile -filetypes $types -title "Open DVI File"]
	if {$name != {}} {
	    if {[catch {set f [dvi::file open $name]} msg]} {
		puts stderr $msg
		return
	    }
	    tkdvi::browser::openWithFile $top $f
	}
    }

    proc close {top} {
	unset tkdvi::browser::$top
	destroy $top
    }

    proc reload {top} {
	upvar \#0 tkdvi::browser::$top state

	dvi::file reload $state(dviFile)
    }

    proc _reload {top} {
	upvar \#0 tkdvi::browser::$top state
	tkdvi::pagesel::addpages $top.f.pgsel \
		[dvi::file pagenumbers $state(dviFile)]

	foreach dvi $state(dvis) {
	    $state($dvi,i) page $state($dvi,p)
	}
    }

    proc checkReload {top} {
	upvar \#0 tkdvi::browser::$top state
	if {[dvi::file changed $state(dviFile)]} {
	    reload $top
	}
    }

    proc clone {top} {
	upvar \#0 tkdvi::browser::$top state
	set t [openWithFile {} $state(dviFile)]
	array set tkdvi::browser::$t [array get state]
    }

    proc openWithFile {top dviFile args} {
	global dvi
	if {[string compare $top {}] == 0} {
	    set count 0
	    set top ".tkdvi[incr count]"
	    while {[winfo exists $top]} {
		set top ".tkdvi[incr count]"
	    }
	}
	catch { destroy $top }
	toplevel $top -class Dvi
	tkdvi::menu::menubar $top.m $top

	dvi::file reloadcmd $dviFile [namespace code [list _reload $top]]

	array set arg {-mode single -size a4 -shrink 8}
	if {[llength $args] % 2 != 0} {
	    error "invalid number of arguments: $args"
	}
	array set arg $args

	upvar \#0 tkdvi::browser::$top state
	set state(dviFile) $dviFile
	set state(mode) $arg(-mode)
	set state(size) $arg(-size)
	set state(shrink) $arg(-shrink)

	wm title $top [format {tkdvi: %s} \
		[file tail [file rootname [lindex [dvi::file info $dviFile] 0]]]]
	$top configure -menu $top.m

	frame $top.f
	label $top.f.posx
	label $top.f.posy
	label $top.f.distance
	pack $top.f.posx $top.f.posy $top.f.distance -side top -fill x

	frame $top.f.page
	button $top.f.page.prev -text "<<<" -padx 5 -pady 2 \
		-command [namespace code [list nextPage $top -1]]
	button $top.f.page.next -text ">>>" -padx 5 -pady 2 \
		-command [namespace code [list nextPage $top 1]]
	pack $top.f.page.prev $top.f.page.next -side left -fill x
	pack $top.f.page -side top -fill x
	
	tkdvi::pagesel::new $top.f.pgsel
	tkdvi::pagesel::addpages $top.f.pgsel \
		[dvi::file pagenumbers $state(dviFile)]
	tkdvi::pagesel::command $top.f.pgsel \
		[namespace code [list goToPage $top %p]]
	pack $top.f.pgsel -side bottom -fill y -expand yes

	frame $top.dvi -borderwidth 2 -relief sunken
	canvas $top.dvi.c -background white \
		-xscrollcommand [list $top.sx set] \
		-yscrollcommand [list $top.sy set]
	scrollbar $top.sx -orient horizontal -command [list $top.dvi.c xview]
	scrollbar $top.sy -orient vertical   -command [list $top.dvi.c yview]

	images $top $top.dvi.c
	trace variable state(mode) w \
		[namespace code [list images $top $top.dvi.c]]
	trace variable state(shrink) w \
		[namespace code [list images $top $top.dvi.c]]
	trace variable state(size) w \
		[namespace code [list images $top $top.dvi.c]]

	set state(resolution) \
		[$state([lindex $state(dvis) 0],i) cget -xresolution]
	set state(unit) cm
	set state(posFmt) {%4.2fcm}
	trace variable state(unit) w \
		[namespace code [list _setPosFmt $top]]

	canvasbindings $top $top.dvi.c

	pack $top.dvi.c -fill both -expand yes
	grid $top.f   -row 0 -column 0 -rowspan 2 -sticky ns
	grid $top.dvi -row 0 -column 1            -sticky nsew
	grid $top.sy  -row 0 -column 2            -sticky ns
	grid $top.sx  -row 1 -column 1            -sticky ew
	grid rowconfigure    $top 0 -weight 1
	grid columnconfigure $top 1 -weight 1

	focus $top.dvi.c
	return $top
    }

    proc images {top canvas args} {
	upvar \#0 tkdvi::browser::$top state

	set startPage 1
	if {[info exists state(dvis)]} {
	    set startPage $state([lindex $state(dvis) 0],p)
	}
	$canvas delete all
	set state(dvis) {}
	if {[string compare $state(mode) single] == 0 \
		|| [string compare $state(mode) spread] == 0} {
	    set state(dvi0,i) [::image create dvi -file $state(dviFile) \
		    -shrink $state(shrink) -size $state(size)]
	    $state(dvi0,i) page $startPage
	    set state(dvi0,x) 0
	    set state(dvi0,y) 0
	    set state(dvi0) [$canvas create image \
		    $state(dvi0,x) $state(dvi0,y) \
		    -image $state(dvi0,i) -anchor nw -tags dvi]
	    set state(dvi0,p) $startPage
	    set state(totalWidth) [image width $state(dvi0,i)]
	    set state(totalHeight) [image height $state(dvi0,i)]
	    lappend state(dvis) dvi0
	}
	if {[string compare $state(mode) spread] == 0} {
	    set state(dvi1,i) [::image create dvi -file $state(dviFile) \
		    -shrink $state(shrink) -size $state(size)]
	    $state(dvi1,i) page [expr {$state(dvi0,p) + 1}]
	    set state(dvi1,x) $state(totalWidth)
	    set state(dvi1,y) $state(dvi0,y)
	    set state(dvi1) [$canvas create image \
		    $state(dvi1,x) $state(dvi1,y) \
		    -image $state(dvi1,i) -anchor nw -tags dvi]
	    set state(dvi1,p) [expr {$state(dvi0,p)+1}]
	    incr state(totalWidth) [image width $state(dvi1,i)]
	    lappend state(dvis) dvi1
	}
	if {[string compare $state(mode) overview] == 0} {
	    set offset 0
	    set yy 0
	    for {set y 0} {$y < 4} {incr y} {
		set xx 0
		for {set x 0} {$x < 4} {incr x} {
		    set dvi dvi$offset
		    set state($dvi,i) [::image create dvi \
			    -file $state(dviFile) -size $state(size) \
			    -shrink [expr {4*$state(shrink)}]]
		    $state($dvi,i) page [expr {$startPage+$offset}]
		    set state($dvi,x) $xx
		    set state($dvi,y) $yy
		    set state($dvi) [$canvas create image \
			    $state($dvi,x) $state($dvi,y) \
			    -image $state($dvi,i) -anchor nw -tags dvi]
		    set state($dvi,p) [expr {$startPage+$offset}]
		    incr xx [image width $state($dvi,i)]
		    incr offset
		    lappend state(dvis) $dvi
		}
		incr yy [image height $state(dvi0,i)]
	    }
	    set state(totalWidth) $xx
	    set state(totalHeight) $yy
	}
	dimensions $top $top.dvi.c
	pagebindings $top $top.dvi.c
	tkdvi::pagesel::markCurrent $top.f.pgsel $state(dvi0,p)
    }

    proc dimensions {top canvas {width 99999} {height 99999}} {
	upvar \#0 tkdvi::browser::$top state
	set wd $state(totalWidth)
	set ht $state(totalHeight)
	if {$wd > $width} { set wd $width }
	if {$ht > $height} { set ht $height }
	$canvas configure -scrollregion [list 0 0 $wd $ht]
	set limit [expr {0.9*[winfo screenwidth $top]}]
	set w [expr {$wd > $limit ? $limit : $wd}]
	set limit [expr {0.9*[winfo screenheight $top]}]
	set h [expr {$ht > $limit ? $limit : $ht}]
	$canvas configure -width $w -height $h
	return [list $wd $ht]
    }

    proc pagebindings {top canvas} {
	upvar \#0 tkdvi::browser::$top state
	set events {<ButtonPress-1> 200 150 \
		<Shift-ButtonPress-1> 400 250 \
		<Control-ButtonPress-1> 700 500}

	foreach item $state(dvis) {
	    foreach {event wd ht} $events {
		$canvas bind $state($item) $event \
			[namespace code \
			[list postMagnifier $top $item %x %y %X %Y $wd $ht 1]]
	    }
	    $canvas bind $state($item) <Enter> \
		    [namespace code [list _showPosition $top $item %x %y]]
	    $canvas bind $state($item) <Motion> \
		    [namespace code [list _showPosition $top $item %x %y]]
	    # $canvas bind $state($item) <Leave> \
	#	    [namespace code [list _hidePosition $top]]
	}
    }

    proc canvasbindings {top canvas} {
	upvar \#0 tkdvi::browser::$top state

	# Finish magnifier bindings
	$canvas bind dvi <B1-Motion> \
		[namespace code [list trackMagnifier $top %x %y %X %Y]]
	$canvas bind dvi <ButtonRelease-1> \
		[namespace code [list unpostMagnifier $top]]

	# Navigation
	bind $canvas <Left> [list $canvas xview scroll -1 pages]
	bind $canvas <Right> [list $canvas xview scroll 1 pages]
	bind $canvas <Down> [list $canvas yview scroll 1 pages]
	bind $canvas <Up> [list $canvas yview scroll -1 pages]

	bind $canvas <minus> [namespace code [list nextPage $top -1]]
	bind $canvas <plus> [namespace code [list nextPage $top 1]]
	bind $canvas <asterisk> [namespace code [list nextPage $top 10]]
	bind $canvas <underscore> [namespace code [list nextPage $top -10]]
	bind $canvas <space> [namespace code [list nextPage $top 1]]

	# Measuring (not in Overview mode)
	set shrink [$state([lindex $state(dvis) 0],i) cget -shrink]
	if {[string compare $state(mode) overview] != 0} {
	    bind $canvas <ButtonPress-3> \
		    [namespace code [list _postMeasure $top %x %y]]
	    bind $canvas <B3-Motion> \
		    [namespace code [list _trackMeasure $top $shrink %x %y]]
	    bind $canvas <ButtonRelease-3> \
		    [namespace code [list _unpostMeasure $top]]
	} else {
	    bind $canvas <ButtonPress-3> {}
	    bind $canvas <B3-Motion> {}
	    bind $canvas <ButtonRelease-3> {}
	}

	# Extras
	bind $canvas <q> { exit }
	bind $canvas <r> [namespace code [list reload $top]]

	bind $top <Expose> [namespace code [list checkReload $top]]
    }

    proc nextPage {top {offset 1}} {
	upvar \#0 tkdvi::browser::$top state
	if {[string compare $state(mode) spread] == 0} {
	    incr offset $offset 
	}
	set lastPage [expr {[lindex [dvi::file info $state(dviFile)] 2]}]
	foreach dvi $state(dvis) {
	    set newPage [expr {$state($dvi,p)+$offset}]
	    if {$newPage > $lastPage} {
		set newPage 1
	    } elseif {$newPage < 1} {
		set newPage $lastPage
	    }
	    $state($dvi,i) page $newPage
	    set state($dvi,p) $newPage
	}
	tkdvi::pagesel::markCurrent $top.f.pgsel $state(dvi0,p)
    }

    proc goToPage {top page} {
	upvar \#0 tkdvi::browser::$top state
	set lastPage [expr {[lindex [dvi::file info $state(dviFile)] 2]}]
	set i 0
	foreach dvi $state(dvis) {
	    # set newPage [expr {($page + $i - 1) % $lastPage + 1}]
	    set newPage [expr {$page + $i}]
	    $state($dvi,i) page $newPage
	    set state($dvi,p) $newPage
	    incr i
	}
	tkdvi::pagesel::markCurrent $top.f.pgsel $state(dvi0,p)
    }	

    proc _showPosition {top item x y} {
	upvar \#0 tkdvi::browser::$top state
	if {[string compare $state(unit) px] == 0} {
	    set shrink 1
	} else {
	    set shrink [$state($item,i) cget -shrink]
	}
	set x [expr {$shrink*([$top.dvi.c canvasx $x]-$state($item,x))}]
	set x [dvi::distance $state(resolution) $x $state(unit)]
	set y [expr {$shrink*([$top.dvi.c canvasy $y]-$state($item,y))}]
	set y [dvi::distance $state(resolution) $y $state(unit)]
	$top.f.posx configure -text [format "x: $state(posFmt)" $x]
	$top.f.posy configure -text [format "y: $state(posFmt)" $y]
    }

    proc postMagnifier {top item x y wx wy width height shrink} {
	upvar \#0 tkdvi::browser::$top state
	catch { destroy $top.mag }
	toplevel $top.mag -borderwidth 2 -relief ridge
	wm overrideredirect $top.mag true
	canvas $top.mag.c -background white -width $width -height $height
	pack $top.mag.c

	wm geometry $top.mag [format {+%d+%d} \
		[expr {$wx-$width/2}] [expr {$wy-$height/2}]]

	set state(mitem) $item
	set state(imgm) [::image create dvi -file $state(dviFile) \
		-shrink $shrink -size $state(size)]
	$state(imgm) page $state($item,p)
	set state(dvim) [$top.mag.c create image 0 0 -image $state(imgm) \
		-anchor nw]
	set x [expr {[$top.dvi.c canvasx $x]-$state($item,x)}]
	set y [expr {[$top.dvi.c canvasy $y]-$state($item,y)}]
	set state(ishrink) [$state($item,i) cget -shrink]
	$top.mag.c move all \
		[expr {-$x*$state(ishrink)+$width/2}] \
		[expr {-$y*$state(ishrink)+$height/2}]
	set state(magX) $x
	set state(magY) $y
	set state(magW) $width
	set state(magH) $height
    }
    proc trackMagnifier {top x y wx wy} {
	upvar \#0 tkdvi::browser::$top state
	set item $state(mitem)
	set x [expr {[$top.dvi.c canvasx $x]-$state($item,x)}]
	set y [expr {[$top.dvi.c canvasy $y]-$state($item,y)}]
	_showPosition $top $item $x $y
	$top.mag.c move all \
		[expr {($state(magX)-$x)*$state(ishrink)}] \
		[expr {($state(magY)-$y)*$state(ishrink)}]
	wm geometry $top.mag [format {+%d+%d} \
		[expr {$wx-$state(magW)/2}] [expr {$wy-$state(magH)/2}]]
	set state(magX) $x
	set state(magY) $y
    }
    proc unpostMagnifier {top} {
	upvar \#0 tkdvi::browser::$top state
	::image delete $state(imgm)
	destroy $top.mag
    }

    proc _postMeasure {top x y} {
	upvar \#0 tkdvi::browser::$top state
	set state(measItem) [$top.dvi.c create line $x $y $x $y -fill black]
	set state(cursor) [$top.dvi.c cget -cursor]
	$top.dvi.c configure -cursor crosshair
	set state(measX) $x
	set state(measY) $y
    }
    proc _trackMeasure {top shrink x y} {
	upvar \#0 tkdvi::browser::$top state
	set coords [$top.dvi.c coords $state(measItem)]
	set coords [lreplace $coords 2 3 $x $y]
	eval $top.dvi.c coords $state(measItem) $coords
	set dx [expr {$state(measX)-$x}]
	set dy [expr {$state(measY)-$y}]
	$top.f.distance configure -text [format "d=$state(posFmt)" \
		[dvi::distance $state(resolution) \
		[expr {$shrink*sqrt($dx*$dx+$dy*$dy)}] \
		$state(unit)] $state(unit)]
    }
    proc _unpostMeasure {top} {
	upvar \#0 tkdvi::browser::$top state
	$top.dvi.c delete $state(measItem)
	$top.dvi.c configure -cursor $state(cursor)
	$top.f.distance configure -text {}
    }

    proc _postShrinkMenu {top menu} {
	upvar \#0 tkdvi::browser::$top state
	$menu delete 0 end
	foreach s {1 2 3 4 6 8 16 32 64} {
	    $menu add radiobutton -label $s \
		    -variable tkdvi::browser::[set top](shrink) -value $s
	}
    }
    proc _postPaperMenu {top menu} {
	upvar \#0 tkdvi::browser::$top state
	if {[llength $state(dvis)] == 0} return
	$menu delete 0 end
	set img $state([lindex $state(dvis) 0],i)
	foreach s [$img defsize] {
	    foreach {w h} [$img defsize $s] break
	    $menu add radiobutton \
	        -label [format {%s (%sx%s)} $s $w $h] \
		    -variable tkdvi::browser::[set top](size) -value $s
	}
    }
    proc _postUnitsMenu {top menu} {
	upvar \#0 tkdvi::browser::$top state
	$menu delete 0 end
	foreach {name unit} {Inch in Centimetres cm Millimetres mm \
		{TeX points} pt {PostScript points} bp Pica pc \
		{Didot points} dd Cicero cc {Scaled points} sp \
	        Pixels px} {
	    $menu add radiobutton \
		    -label [format {%s (%s)} $name $unit] \
		    -variable tkdvi::browser::[set top](unit) -value $unit
	}
    }

    variable _unitFmts
    array set _unitFmts {
	in "%4.2fin"
	cm "%4.2fcm"
	mm "%5.1fmm"
	pt "%6.1fpt"
	bp "%6.1fbp"
	pc "%5.1fpc"
	dd "%6.1fdd"
	cc "%5.1fcc"
	sp "%8.0fsp"
	px "%4.0fpx"
    }

    proc _setPosFmt {top name1 name2 op} {
	variable _unitFmts
	upvar \#0 tkdvi::browser::$top state
	upvar [set name1]($name2) unit
	puts stderr "$unit"
	set state(posFmt) $_unitFmts($unit)
    }

    proc print {top {pages {}}} {
	upvar \#0 tkdvi::browser::$top state
	set fileName [lindex [dvi::file info $state(dviFile)] 0]
	if {$pages != {}} {
	    set pages "-pp [join $pages ,]"
	}
	# TODO: Put up a fancy dialog here.
	set cmd [format {dvips %s %s} $pages $fileName]
	eval exec $cmd >& /dev/null
    }

    proc printMarked {top} {
	upvar \#0 tkdvi::browser::$top state
	print $top [tkdvi::pagesel::selected $top.f.pgsel]
    }

    proc about {} {
	global tkdvi aboutOK
	set top [tkdvi::dialog::new {} About]
	set c [tkdvi::dialog::content $top]
	set b [tkdvi::dialog::buttons $top]
	set i [image create photo -file [file join $tkdvi(tcllib) tkdvi.gif]]
	label $c.icon -image $i
	pack $c.icon -padx 5 -pady 5 -side left
	set text "TkDVI, a Tcl/Tk DVI Previewer\n"
	append text "Version [tkdvi::version]\n"
	append text "Copyright  1999 Anselm Lingnau\n\n"
	append text "For information, see\n"
	append text "http://tm.informatik.uni-frankfurt.de/~lingnau/tkdvi/"
	message $c.msg -aspect 500 -text $text
        pack $c.msg -padx 5 -pady 5 -side right -expand yes -fill both
	button $b.ok -text OK -command {set aboutOK 1}
	pack $b.ok -side left -expand yes
	focus $b.ok
	wm protocol $top WM_DELETE_WINDOW [list $b.ok invoke]
	tkdvi::dialog::wait $top aboutOK
	destroy $top
    }
}
