/*
 * dlh - a LaTeX to HTML translator.
 * Copyright (C) 1996 David Mosberger-Tang.
 * This file is part of the dlh package.
 *
 * The dlh translator is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The dlh translator is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with dlh; see the file COPYING.  If not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "dlh.h"
#include "dlh_math.h"

struct sh *	mtab;

static int	suppress_eqno;


void
math_next_column (void)
{
    /* we can't really emulate this... */
}


/*
 *  _TEXARG
 * or
 *  ^TEXARG
 *
 * Parsing a TeX argument is non-trivial and not a goal of dlh.
 * Unfortunately, enforcing the LaTeX syntax that requires curly
 * braces around the sub/super-script arguments would be painful since
 * there is probably tons of LaTeX code out there with formulas that
 * read $c^2=a^2+b^2$.  So, as a compromise, a TEXARG in dlh may
 * either be an arbitrary argument in curly braces or a single
 * (non-backslash) character immediately after the script-shift
 * command.
 */
void 
math_script (int up)
{
    static const char * html_cmd[] = {
	"sub", "sup"
    };
    int ch = *pos;
    const char * cmd = html_cmd[up];

    switch (ch) {
      case '\\':
	  fprintf(stderr,
		  _("%s:%lu:warning: must enclose argument in curly braces\n"),
		  filename, linenum);
	  return;

      case '{':
	  push_env(&anonymous_env);
	  select_font(cmd);
	  break;

      default:
	  fprintf(out->file, "<%s>%c</%s>", cmd, ch, cmd);
	  break;
    }
    ++pos;
}


/*
 * \begin{math}
 * \end{math}
 */
void
env_math (struct call * c, int prologue)
{
    if (prologue) {
	aux_flags |=  AUX_MATH_MODE;
    } else {
	aux_flags &= ~AUX_MATH_MODE;
    }
}


/*
 * \begin{displaymath}
 * \end{displaymath}
 */
void
env_displaymath (struct call * c, int prologue)
{
    if (prologue) {
	aux_flags |=  AUX_MATH_MODE;
	fputs("\n<p>\n<center>\n", out->file);
    } else {
	aux_flags &= ~AUX_MATH_MODE;
	fputs("</center>\n<p>", out->file);
    }
}


/*
 * \begin{eqnarray}
 * \end{eqnarray}
 *
 * \begin{eqnarray*}
 * \end{eqnarray*}
 *
 * \begin{equation}
 * \end{equation}
 */
void
env_eqnarray (struct call * c, int prologue)
{
    if (prologue) {
	aux_flags |=  AUX_MATH_MODE;
	fputs("\n<p>\n<table width=\"95%\"><tr><td align=center>",
	      out->file);
    } else {
	aux_flags &= ~AUX_MATH_MODE;
	fputs("\n</td><td align=right>", out->file);
	if (strcmp (c->u.env->name, "eqnarray*") != 0) {
	    int num = ++counter[CNT_EQUATION];

	    fprintf(out->file, "(%d)<a name=\"e%d\">", num, num);
	}
	fputs("</td></tr></table>\n<p>\n", out->file);
    }
}

static void
cmd_math_nonumber (struct call * c)
{
    ++suppress_eqno;
}

static void
cmd_math_line_break (struct call * c)
{
    fprintf(out->file, "\n</td><td align=right>");
    if (env && env->e && strcmp(env->e->name, "eqnarray*") != 0) {
	if (suppress_eqno) {
	    --suppress_eqno;
	} else {
	    fprintf(out->file, "(%d)", ++counter[CNT_EQUATION]);
	}
    }
    fputs("</td></tr><tr><td align=center>\n", out->file);
}


static struct cmd math_list[] = {
    {name: "Leftarrow",		       op: cmd_emit,  arg: "&lt;="},
    {name: "Leftarrow",		       op: cmd_emit,  arg: "&lt;="},
    {name: "Rightarrow",	       op: cmd_emit,  arg: "=&gt;"},
    {name: "\\",	pat: ARG_OPT1, op: cmd_math_line_break},
    {name: "cdot",		       op: cmd_emit,  arg: "*"},
    {name: "emptyset",		       op: cmd_emit,  arg: "&Oslash;"},
    {name: "frac",	pat: ARG_REQ2, op: cmd_macro, arg: "((#1)/(#2))"},
    {name: "ge",		       op: cmd_emit,  arg: "&gt;="},
    {name: "geq",		       op: cmd_emit,  arg: "&gt;="},
    {name: "gg",		       op: cmd_emit,  arg: "&gt;&gt;"},
    {name: "langle",		       op: cmd_emit,  arg: "&lt;"},
    {name: "le",		       op: cmd_emit,  arg: "&lt;="},
    {name: "leftarrow",		       op: cmd_emit,  arg: "&lt;-"},
    {name: "leq",		       op: cmd_emit,  arg: "&lt;="},
    {name: "ll",		       op: cmd_emit,  arg: "&lt;&lt;"},
    {name: "mathbf",	pat: ARG_REQ1, op: cmd_macro, arg: "{\\bf #1}"},
    {name: "mathcal",	pat: ARG_REQ1, op: cmd_macro, arg: "#1"}, /* lossage */
    {name: "mathit",	pat: ARG_REQ1, op: cmd_macro, arg: "{\\it #1}"},
    {name: "mathrm",	pat: ARG_REQ1, op: cmd_macro, arg: "#1"}, /* lossage */
    {name: "mathsf",	pat: ARG_REQ1, op: cmd_macro, arg: "{\\sf #1}"},
    {name: "mathtt",	pat: ARG_REQ1, op: cmd_macro, arg: "{\\tt #1}"},
    {name: "mu",		       op: cmd_emit,  arg: "&micro;"},
    {name: "nonumber",		       op: cmd_math_nonumber},
    {name: "pm",		       op: cmd_emit,  arg: "&plusmn;"},
    {name: "rangle",		       op: cmd_emit,  arg: "&gt;"},
    {name: "rightarrow",	       op: cmd_emit,  arg: "-&gt;"},
    {name: "times",		       op: cmd_emit,  arg: " &times; "},
    {name: "wedge",		       op: cmd_emit,  arg: "^"}
};


void
math_init (void)
{
    int i;

    /* enter all math commands in hash-table: */
    mtab = sh_new(1024);
    for (i = 0; i < NELEMS(math_list); ++i) {
	sh_enter(mtab, math_list[i].name, math_list + i);
    }
}
