/* SIMPLE - Simple Is a Macro Processing Language Element */
/* Copyright (c) 1998 David A. Madore (david.madore@ens.fr) */

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * This file takes care of expanding tokens; it is in many ways the
 * central module.
 */

#include <stdio.h>
#include <stdlib.h>

#include "simple.h"
#include "token.h"

void
ship_out(token_list *tl,token tk)
     /* Append the token tk to the token list tl.  If tl==NULL then
      * output the token. */
{
  if (!tl) output_token(tk);
  else append_token(tl,tk);
}

static token_list input_list={NULL,0,0};
/* This contains tokens that were generated by macro expansion and are
 * to be reprocessed. */

token
ship_in(void)
     /* Read a token from the input list or the input file if the
      * input list is empty. */
{
  token tk;
  if (is_empty(&input_list))
    return input_token();
  else {
    tk=first_token(&input_list);
    remove_first(&input_list);
    return tk;
  }
}

void expand_token(token_list *tl,token tk);

void
expand_args(int *argc,token_list **argv)
     /* Read in the arguments to a macro and expand them.  Return in
      * argc the number of arguments and in argv a malloc()ated array
      * of token lists corresponding to the various arguments (after
      * expansion of course). */
{
  token tk;
  *argc=1;
  *argv=malloc(sizeof(token_list));
  while (1) {
    make_empty_list(*argv+(*argc-1));
    while (1) {
      tk=ship_in();
      if (tk==EOF)
	fatal(0,"Unterminated command.");
      if ((tk==END_COMMAND)||(tk==NEXT_PARAM))
	break;
      expand_token(*argv+(*argc-1),tk);
    }
    if (tk==END_COMMAND)
      break;
    (*argc)++;
    if ((*argv=REALLOC(*argv,*argc*sizeof(token_list)))==NULL)
      fatal(0,"Out of memory!");
  }
}

void
call_macro(token_list *expansion,int argc,token_list *argv)
     /* Call a (user or builtin) macro, with argc arguments given by
      * argv (of course, argv[0] is the macro name itself), and put
      * its expansion in expansion. */
{
  char *name;
  command cmd;
#if 0
  if (argc<1)
    fatal(1,"argc<1");
#else
  if (argc<1)
    return;
  /* Void call is not an error; it is just ignored. */
#endif
  name=token_list_to_string(argv);
  cmd=identify_command(name);
  FREE(name);
  if (cmd<0)
    call_builtin(expansion,cmd,argc,argv);
  else
    call_user(expansion,cmd,argc,argv);
}

void
expand_macro(void)
     /* Read the arguments to a macro and expand it, putting the
      * expansion back on the input list. */
{
  int argc;
  token_list *argv;
  token_list expansion;
  expand_args(&argc,&argv);
  make_empty_list(&expansion);
  call_macro(&expansion,argc,argv);
  for (;argc>0;argc--) free_list(argv+argc-1);
  FREE(argv);
  prepend_list(&input_list,&expansion);
  free_list(&expansion);
}

void
expand_token(token_list *tl,token tk)
     /* Expand the token tk to the token list tl.  If tl==NULL, expand
      * to the output. */
{
  int quote_level;
  if ((tk==EOF)||(tk<FIRST_SPECIAL))
    ship_out(tl,tk);
  else if (tk==QUOTE_NEXT) {
    tk=ship_in();
    ship_out(tl,tk);
  } else if (tk==OPEN_QUOTE) {
    quote_level=1;
    while (quote_level>=1) {
      tk=ship_in();
      if (tk==EOF)
	fatal(0,"Unterminated quotation.");
      else if (tk==OPEN_QUOTE)
	quote_level++;
      else if (tk==CLOSE_QUOTE)
	quote_level--;
      else if (tk==QUOTE_NEXT)
	tk=ship_in();
      if (quote_level>=1)
	ship_out(tl,tk);
    }
  } else if (tk==BEGIN_COMMAND)
    expand_macro();
  /* Note that macro expansion goes to the INPUT LIST not to the tl
   * token list, so it will be reread. */
  else if (tk==AT_SIGN)
    ship_out(tl,tk);
  /* At sign gets shipped out so we don't have to put quote characters
   * around every single damn macro. Of course, if it finds its way to
   * the output, it gets kicked out at that point. */
  else
    fatal(0,"Misplaced special token %d.",tk);
}

void
expand_input(void)
     /* This is the main program loop. */
     /* Read the input and expand tokens as we go. */
{
  while (1)
    expand_token(NULL,ship_in());
}
