## -*-Tcl-*-
 # ###################################################################
 #  AlphaTk - the ultimate editor
 # 
 #  FILE: "alpha_menus.tcl"
 #                                    created: 04/12/98 {23:17:46 PM} 
 #                                last update: 1999-09-05T19:32:57Z 
 #  Author: Vince Darley
 #  E-mail: vince@santafe.edu
 #    mail: 317 Paseo de Peralta, Santa Fe, NM 87501
 #     www: http://www.santafe.edu/~vince
 #  
 # Copyright (c) 1998-1999  Vince Darley
 # 
 # See the file "license.terms" for information on usage and redistribution
 # of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 #  Description: 
 # 
 #  History
 # 
 #  modified by  rev reason
 #  -------- --- --- -----------
 #  04/12/98 VMD 1.0 original
 # ###################################################################
 ##
#==============================================================================
#= Alpha Commands
#==============================================================================
#
#In this list of routines, text between '<' and '>' is a placeholder for a 
#required parameter, text between '[' and ']' is a placeholder for an 
#optional parameter, and the '|' signifies a choice of two or more 
#alternatives.  A '+' signifies that the previous symbol can be present one 
#or more times, while a '*" means zero or more times.  Some commands have no 
#parameters, and are only expected to be called interactively.
#
#
#
#The following are Alpha-specific tcl routines:
#

 
#  menus  #

# addMenuItem [-m] [-l <meta-characters>] <menu name> <item name> - Convert
#  item to menu form and add to specified menu. If '-m' specified, menu 
#  form conversion not done. The '-l' option allows you to use menu meta 
#  characters as text of menu items. If the '-l' option is used, the 
#  <meta-characters> string is interpreted for menu item attributes, and 
#  meta-characters in <item name> are included in the menu item text 
#  literally. For example: 
#  	addMenuItem -m -l "/B" "Hello/C" 
#  would appear as the text "Hello/C" in the menu, and have "B" as its 
#  command equivalent.
proc addMenuItem {name item args} {
    # doesn't currently take account of the proc attached to the given
    # menu.
    set accel ""
    if {[llength $args]} {
	if {$name == "-m"} { 
	    set mflag 1
	    if {$item == "-l"} {
		foreach {accel name item index} $args {}
	    } else {
		set name $item 
		set item [lindex $args 0]
		set index [lindex $args 1]
	    }
	} elseif {$name == "-l"} {
	    set accel $item
	    foreach {name item index} $args {}
	} else {
	    set index [lindex $args 0]
	}
	if {$index == ""} {
	    set index "add"
	} else {
	    set index [list insert $index]
	}
    }
    global alpha_mprocs
    if {[info exists alpha_mprocs($name)]} {
	set mproc $alpha_mprocs($name)
    } else {
	set mproc "menu::generalProc"
    }
    set name [menu_name $name]
    regsub -all " " $name "_" n
    set n m$n
    if {$item == "\(-"} {
	eval [list .menubar.$n] $index separator
    } else {
	if {[info exists mflag]} { 
	    set label $item
	} else {
	    set label [quote::Menuify $item]
	}
	regsub {$} $label "..." label
	regsub {} $label ">>" label
	if {$accel != ""} {
	    foreach {accelerator accel} [alpha::accelerator [string range $accel 1 end] ""] {}
	    set accel [list -accelerator $accel]
	    if {$accelerator != ""} {
		regsub -all "Ctrl" $accelerator "Control" accelerator
		bind all "<${accelerator}>" [list $mproc $name $item]
	    }
	}
	eval [list .menubar.$n] $index \
	  command -label [list [::msgcat::mc $label]] \
	  -command [list [list $mproc $name $item]] \
	  $accel
    }
    
}

proc addHelpMenu {item} {
    eval .menubar.help add [alpha::menuItem $item "helpMenu"]
    global multiColumnMenusEveryNItems
    if {[.menubar.help index end] == $multiColumnMenusEveryNItems} {
	.menubar.help entryconfigure end -columnbreak 1
    }
}
proc alpha::menuItem {item {proc ""}} {
    if {$item != "(-"} {
	return [list command -label [::msgcat::mc $item] -command "$proc [list $item]"]
    } else {
	return "separator"
    }
}

proc alpha::accelerator {key mods} {
    variable command_key 
    variable option_key
    variable keymap
    variable menukeymap
    if {$key == ""} {return ""}
    regsub -all {<[SE]} $mods "" mods
    if {$mods != ""} {
	regsub "<I" $mods "${option_key}-" mods
	regsub "<U" $mods "Shift-" mods
	regsub "<B" $mods "Ctrl-" mods
	regsub "<O" $mods "${command_key}-" mods
    }
    if {[regexp {[a-z]} $key]} {
	global keys::func tcl_platform
	# Enter is invalid on non-MacOS 
	if {$key == "a" && ($tcl_platform(platform) != "macintosh")} {
	    return ""
	}
	set rest [lindex $keys::func [expr {[text::Ascii $key] -97}]]
	if {$rest == "Delete"} {
	    set rest BackSpace
	} elseif {$rest == "Fwd Del"} {
	    set rest Delete
	}
	set menu $mods$rest
	append mods KeyPress-$rest
    } else {
	if {![info exists alpha::menukeymap($key)]} {
	    if {$mods == "" || $mods == "Shift-"} { 
		append mods $command_key "-" 
	    }
	    set menu $mods$key
	    if {[info exists alpha::keymap($key)]} {
		set key $alpha::keymap($key)
	    }
	} else {
	    set menu $mods$key
	    set key $alpha::menukeymap($key)
	}
	append mods KeyPress- $key
    }
    return [list $mods $menu]
}

# deleteMenuItem [-m] <menu name> <item name> - Convert item to menu form 
#  and delete from specified menu. If '-m' specified, menu form conversion 
#  not done. 
proc deleteMenuItem {name item args} {
    if {[llength $args]} {
	if {$name == "-m"} { 
	    set mflag 1
	    set name $item 
	    set item [lindex $args 0]
	}
    }
    regsub -all " " [menu_name $name] "_" n
    set n m$n
    regsub {$} $item "..." item
    regsub {} $item ">>" item
    if {[info exists mflag]} { 
	set label $item
    } else {
	set label [quote::Menuify $item]
    }
    .menubar.$n delete [.menubar.$n index [::msgcat::mc $label]]
}

proc renameMenuItem {name item newitem args} {
    if {[llength $args]} {
	if {$name == "-m"} { 
	    set mflag 1
	    set name $item 
	    set item $newitem
	    set newitem [lindex $args 0]
	}
    }
    regsub -all " " [menu_name $name] "_" n
    set n m$n
    regsub {$} $item "..." item
    regsub {} $item ">>" item
    regsub {$} $newitem "..." newitem
    regsub {} $newitem ">>" newitem
    if {[info exists mflag]} { 
	set label $item
    } else {
	set label [quote::Menuify $item]
	set newitem [quote::Menuify $newitem]
    }
    .menubar.$n entryconfigure [::msgcat::mc $label] -label [::msgcat::mc $newitem]
}

# enableMenuItem <menuName> <item text> <on|off> - Either enable or 
#  disable the menu item of user menu 'menuName' that has text '<item 
#  text>'. Note that unless the menu is not only created, but also already
#  inserted, this command has no effect. 
proc enableMenuItem {name item on args} {
    if {[llength $args]} {
	if {$name == "-m"} {
             set mflag 1
	     set name $item
             set item $on
	     set on [lindex $args 0]
        }
    }
    
    regsub -all " " [menu_name $name] "_" n
    set n m$n
    if {![winfo exists .menubar.$n]} { error "No such menu $name" }
    if {$item == ""} {
	# it's the whole menu
	.menubar entryconfigure [menu_name $name] \
	  -state [expr {$on ? "normal" : "disabled"}]
	return
    }

    regsub {$} $item "..." item
    regsub {} $item ">>" item
    
    if {[info exists mflag]} {
	catch {.menubar.$n entryconfigure [::msgcat::mc $item] \
	  -state [expr {$on ? "normal" : "disabled"}]}
    } else {
	catch {.menubar.$n entryconfigure [::msgcat::mc [quote::Menuify $item]] \
	  -state [expr {$on ? "normal" : "disabled"}]}
    }
    
}

# insertMenu <name> - insert the previously created user menu 'name' into 
#  the menuBar. 
proc insertMenu {n} {
    set n [menu_name $n]
    regsub -all " " $n "_" w
    set w m$w
    if {![winfo exists .menubar.$w]} {
	menu .menubar.$w
    } 
    if {[catch {.menubar index $n}]} {
	.menubar insert Help cascade -label "$n" -menu .menubar.$w
    }
}
# markMenuItem [-m] <menuName> <item text> <on|off> [<mark char>] - Either mark or unmark
#  the menu item of user menu 'menuName' that has text '<item text>'. 
#  Note that unless the menu is not only created, but also already
#  been inserted, this command has no effect. 
proc markMenuItem {m item on {char ""} args} {
    if {$m == "-m"} {
	set m $item ; set item $on ; set on $char ; set char [lindex $args 0]
    } else {
	set item [quote::Menuify $item]
    }
    regsub {$} $item "..." item
    regsub {} $item ">>" item
    set m [menu_name $m]
    regsub -all " " $m "_" m
    set m .menubar.m$m
    if {$on == "on"} { set on 1 } elseif {$on == "off"} { set on 0 }
    if {[catch {$m index $item} index]} {
	return ""
    }
    set label [$m entrycget $index -label]
    global alphaDummy
    switch -- [$m type $index] {
	"radiobutton" {
	    if {$on} {
		set alphaDummy(menu,$m,$label) $label
	    }
	}
	"checkbutton" {
	    set alphaDummy(menu,$m,$label) $on
	}
	default {
	    # this is so the user can create the item as an ordinary
	    # menu item, and then later turn it into a checkbutton
	    # without any problems.
	    $m insert $index checkbutton \
	      -command [$m entrycget $index -command] \
	      -label [::msgcat::mc $label] \
	      -accelerator [$m entrycget $index -accelerator] \
	      -columnbreak [$m entrycget $index -columnbreak] \
	      -state normal \
	      -variable alphaDummy(menu,$m,$label)
	    set alphaDummy(menu,$m,$label) $on
	    $m delete [expr {$index +1}]
	}
    }
    return ""
}

# Menu [-s] [-n <name>] [-i <num] [-m] [-M <mode>] [-p <procname>] <list of menu items> - 
#  Defines a new menu named 'name' (if provided w/ '-n' option). The menu is not 
#  yet inserted into the menubar. The menu commands may be nested for 
#  heirarchical menus, see 'AlphaBits.tcl' for examples. Alpha massages the 
#  function names to make them look better in the menus. 
#  '-c'			Ignore any menu meta-chars. Can also be done on a per-item basis 
#  				by appending an ampersand ('&') to the end of an item.
#  '-s'			Use system menu definition proc (faster).
#  '-n <num>'    Resource id of icon suite to use for menu title. 'ics#' 
#                is the only resource that is really necessary.
#  '-n <name>'	Name the menu. Not necessary for submenus.
#  '-m'			No menu form. If not supplied, each menu item is split into 
#  				words at each capitalized letter.
#  '-p <name>' 	The tcl proc named by 'name' is called w/ the menu's name
#  				and the item's contents when the item is chosen.
#  '-M <mode>'	Specifies that any bindings created by the menu are 
#  				specific to a given mode. This is important because mode-specific
#  				bindings over-ride global bindings.
proc Menu {args} {
    set ma [lindex $args end]
    set args [lreplace $args end end]
    getOpts {-n -M -p -t -h}
    if {[info exists opts(-p)]} {
	lappend proc $opts(-p)
	global alpha_mprocs
	set alpha_mprocs($opts(-n)) $proc
    } else {
	set proc ""
    }
    #if {[info exists opts(-M)]} { lappend proc -M $opts(-m) }
    #if {[info exists opts(-m)]} { lappend proc -m }
    set opts(-n) [menu_name $opts(-n)]
    regsub -all " " $opts(-n) "_" m
    set m .menubar.m$m
    if {[winfo exists $m]} {
	destroy $m
    }
    global tearoffMenus
    menu $m -tearoff $tearoffMenus
    
    global multiColumnMenusEveryNItems
    set count [expr {$multiColumnMenusEveryNItems +1}]
    foreach item $ma {
	incr count -1
	# special 'catch' so we don't trip on 'lindex " 0'
	if {![catch {lindex $item 0} tmp] && $tmp == "Menu"} {
	    foreach {mm label} [eval Menu [lrange $item 1 end]] {}
	    if {![info exists opts(-m)]} {
		set label [quote::Menuify $label]
	    }
	    regsub {$} $label "..." label
	    regsub {} $label ">>" label
	    $m add cascade -label [::msgcat::mc $label] -menu $mm

	} elseif {[info exists opts(-c)]} {
	    # ignore all meta characters
	    if {$item == "\(-"} {
		$m add separator
		continue
	    } else {
		set state "normal"
		set entrytype command
		set isOn 0
		set label $item

		if {![info exists opts(-m)]} {
		    set label [quote::Menuify $label]
		} 
		regsub {$} $label "..." label
		regsub {} $label ">>" label
		regexp {^(.*)$} $item "" item
		if {[info exists opts(-t)]} {
		    set entrytype $opts(-t)
		}
		if {$proc == ""} {
		    $m add $entrytype -label [::msgcat::mc $label] \
		      -command "$item" -state $state
		} else {
		    $m add $entrytype -label [::msgcat::mc $label] \
		      -command "$proc [list $opts(-n) $item]" \
		      -state $state
		}
		if {[info exists opts(-t)]} {
		    global alphaDummy
		    if {$opts(-t) == "radiobutton"} {
			if {$isOn} {
			    set alphaDummy(menu,$m,$label) $label
			}
			$m entryconfigure [::msgcat::mc $label] -variable \
			  alphaDummy(menu,$m,$label) -value $label
		    } else {
			set alphaDummy(menu,$m,$label) $isOn
			$m entryconfigure [::msgcat::mc $label] -variable \
			  alphaDummy(menu,$m,$label)
		    }
		    
		}
	    }
	} else {
	    switch -regexp -- $item {
		"/." {
		    regexp {/(.)} $item "" key
		    regsub "/[quote::Regfind ${key}]" $item "" item 
		    if {$key == "\x1e"} {
			# special case 'icon'
			set key ""
		    }
		    regexp {^((<[UIOCSEB])*)} $item "" mods
		    set item [string range $item [string length $mods] end]					
		    regexp {^(.*)(\\?&|\^.)$} $item "" item
		    set state [expr {[regexp {\((.*)} $item "" item] ? \
		      "disabled" : "normal"}]
			
		    if {![info exists opts(-m)]} {
			set label [quote::Menuify $item]
		    } else {
			set label $item
		    }
		    regsub {$} $label "..." label
		    regsub {} $label ">>" label
		    regexp {^(.*)$} $item "" item
		    
		    if {$proc == ""} {
			set cmd $item
		    } else {
			set cmd "$proc [list $opts(-n) $item]"
		    }
		    set accelerator ""
		    set accel ""
		    foreach {accelerator accel} [alpha::accelerator $key $mods] {}
		    $m add command -label [::msgcat::mc $label] -command $cmd \
		      -accelerator $accel -state $state
		    
		    if {$accelerator != ""} {
			regsub -all "Ctrl" $accelerator "Control" accelerator
			if {[regexp {[A-Z]$} $accelerator last]} {
			    regsub {[A-Z]$} $accelerator [string tolower $last] accelerator
			}
			set to "Alpha"
			#set to "AlphaMenu"
			if {[info exists opts(-M)]} {
			    set to "$opts(-M)AlphaStyle"
			    #set to "$opts(-M)AlphaMenuStyle"
			}
			
			bind $to "<${accelerator}>" "[list $m invoke $label] ; break"
		    }
		}
		{\(-} {
		    $m add separator
		    continue
		}
		default {
		    regsub {^(<[UIOCSEB])*} $item "" item
		    regexp {^(.*)(&|\^.)$} $item "" item
		    set state [expr {[regexp {\((.*)} $item "" item] ? \
		      "disabled" : "normal"}]
		    set entrytype command
		    set isOn 0
		    if {[regexp {^!(.)} $item "" markc]} {
			switch -- $markc {
			    "" {
				set entrytype checkbutton
				set item [string range $item 2 end]
				set label $item
				set isOn 1
			    }
			    " " {
				set entrytype checkbutton
				set item [string range $item 2 end]
				set label $item
				set isOn 0
			    }
			    default {
				set item [string range $item 2 end]
				set label "$markc   $item"
			    }
			}
		    } else {
			set label $item
		    }
		    if {![info exists opts(-m)]} {
			set label [quote::Menuify $label]
		    } 
		    regsub {$} $label "..." label
		    regsub {} $label ">>" label
		    regexp {^(.*)$} $item "" item
		    if {[info exists opts(-t)]} {
			set entrytype $opts(-t)
		    }
		    if {$proc == ""} {
			$m add $entrytype -label [::msgcat::mc $label] \
			  -command "$item" -state $state
		    } else {
			$m add $entrytype -label [::msgcat::mc $label] \
			  -command "$proc [list $opts(-n) $item]" \
			  -state $state
		    }
		    if {[info exists opts(-t)]} {
			global alphaDummy
			if {$opts(-t) == "radiobutton"} {
			    if {$isOn} {
				set alphaDummy(menu,$m,$label) $label
			    }
			    $m entryconfigure [::msgcat::mc $label] -variable \
			      alphaDummy(menu,$m,$label) -value $label
			} else {
			    set alphaDummy(menu,$m,$label) $isOn
			    $m entryconfigure [::msgcat::mc $label] -variable \
			      alphaDummy(menu,$m,$label)
			}
			
		    }
		}
	    }
	    # that was the end of the switch

	}
	# that was the end of the if 'cascade' else 'switch'
	if {$count < 0} {
	    $m entryconfigure [::msgcat::mc $label] -columnbreak 1
	    set count $multiColumnMenusEveryNItems
	} 
    }
    return [list $m $opts(-n)]
}
# removeMenu <name> - remove menu 'name' from menubar, except those 
#  specified by previous 'makeMenuPermanent' calls.
proc removeMenu {n} {
    catch {.menubar delete [menu_name $n]}
}

proc menu_name {n} {
    if {[regexp {^[\245]} $n]} {
	global menuFunnynames index::feature
	if {[info exists menuFunnynames($n)]} { return $menuFunnynames($n) }
	if {[info exists index::feature]} {
	    foreach m [array names index::feature] {
		if {![catch [list uplevel \#0 [list set $m]] res]} {
		    if {$res == $n} {
			regexp {(.*)Menu$} $m "" menuFunnynames($n)
			return [::msgcat::mc [set menuFunnynames($n) [quote::Menuify $menuFunnynames($n)]]]
		    }
		}
	    }
	    set menuFunnynames($n) " Windows"
	    return [::msgcat::mc " Windows"]
	}
    } else {
	regsub -all {\.} $n {} n
	return [::msgcat::mc $n]
    }
}

#  bindings  #

# deleteModeBindings <mode> - Delete all bindings for specified mode.
proc deleteModeBindings {args} {echo "deleteModeBindings $args"}
# describeBinding - display the binding of the next typed key sequence. 
proc describeBinding {} {
    set key [alpha::waitForKey]
    global mode
    foreach tag [list ${mode}AlphaStyle AlphaStyle Alpha ${mode}AlphaMenuStyle AlphaMenuStyle AlphaMenu] {
	if {[bind $tag "<${key}>"] != ""} {
	    alertnote "$key bound to [bind $tag <${key}>]"
	    return
	}
    }
    alertnote "No binding for $key!"
}
array set alpha::mods [list Ctrl 144 Control 144 Shift 34 Option 72 \
  Cmd 1 Command 1 Meta 256 Alt 512]
set alpha::modifiers [list Ctrl Control Shift Option Alt Command Cmd Meta]
proc alpha::waitForKey {} {
    global alphaPriv
    variable modifiers
    set oldFocus [focus]
    set oldGrab [grab current .status]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    message "Press any key combination."
    grab -global .status
    bind .status <KeyPress> {set alphaPriv(done) [list %A %K %N]}
    focus .status

    while 1 {
	vwait alphaPriv(done)
	echo $alphaPriv(done)
	regsub -all -- {_[LR]} [set keycode [lindex $alphaPriv(done) 1]] "" keycode
	append key "-" $keycode
	if {[lsearch -exact $modifiers $keycode] == -1} {
	    break
	}
    }
    
    unset alphaPriv(done)
    bind .status <KeyPress> ""

    catch {grab release .status}
    catch {focus $oldFocus}
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }
    regsub -all {_[LR]} $key "" key
    return [string range $key 1 end]
}

array set alpha::asciikeymap [list 0x20 " " 0x08 0x33 0x7f deleteChar]

# ascii (see bindings).
proc ascii {key args} {
    global alpha::asciikeymap
    if {[info exists alpha::asciikeymap($key)]} {
	set key $alpha::asciikeymap($key)
    } else {
	set key [text::Ascii $key 1]
    }
    uplevel Bind [list $key] $args
}
# unascii  (see bindings)
proc unascii {key args} {
    global alpha::asciikeymap
    if {[info exists alpha::asciikeymap($key)]} {
	set key $alpha::asciikeymap($key)
    } else {
	set key [text::Ascii $key 1]
    }
    uplevel unBind [list $key] $args
}

# bindingList - list all current bindings.
proc bindingList {} {
    echo "bindingList not perfect yet"
    global mode
    set tags [list ${mode}AlphaStyle AlphaStyle Alpha ${mode}AlphaMenuStyle AlphaMenuStyle AlphaMenu]
    set res {}
    foreach tag $tags {
	foreach b [bind $tag] {
	    set fn [bind $tag $b]
	    lappend res [list Bind $tag $b $fn]
	}
    }
    return [join $res "\r"]
}

# unBind  (see bindings)
proc unBind {key mods args} {
    # blank out the script and send it to Bind to be zapped.
    if {[string index $mods 0] == "<"} {
	set args [lreplace $args 0 0 ""]
    } else {
	set mods ""
    }
    
    eval [list Bind $key $mods] $args
}

# bind  (see bindings)
proc Bind {key mods args} {
    global alpha::keymap alpha::command_key alpha::option_key \
      tcl_platform alpha::menukeymap
    switch -- [llength $args] {
	0 { 
	    set script $mods 
	    set mods "" 
	    set bmode "" 
	}
	1 {
	    if {[string index $mods 0] == "<"} {
		set script [lindex $args 0]
		set bmode ""
	    } else {
		set script $mods
		set mods ""
		set bmode [lindex $args 0]
	    }
	}
	2 { 
	    set script [lindex $args 0]
	    set bmode [lindex $args 1] 
	}
	default {
	    error "Too many args to 'Bind'"
	}
    }
    set bind "<"
    if {[regexp {s} $mods]} {append bind "Shift-"}
    if {[regexp {z} $mods]} {append bind "Control-"}
    if {[regexp {o} $mods]} {
	append bind $alpha::option_key "-"
	set have_option 1
    }
    if {[regexp {c} $mods]} {append bind $alpha::command_key "-"}
    
    regexp "'(.)'" $key "" key
    if {[string length $key] > 1 && [regexp {^[a-z]} $key] \
      && ($key != "space") && ($key != "enter")} {
	set key "[string toupper [string index $key 0]][string range $key 1 end]"
    }
    # we don't deal with Key-pad keys separately.
    if {[string first Kpad $key] != -1} {
	return
    }
    if {[info exists alpha::keymap($key)]} {
	set key $alpha::keymap($key)
    } elseif {[info exists alpha::menukeymap($key)]} {
	set key $alpha::menukeymap($key)
    }
    append bind "KeyPress-" $key ">"
    if {$script == "prefixChar"} {
	set script "prefixChar ${bmode}Prefix-[string toupper $key]"
    } elseif {$script == "startEscape"} {
	set script "prefixChar ${bmode}Prefix-e"
    }
    if {[regexp {[eA-Z]} $mods prefix]} {
	append bmode Prefix- $prefix
	# auto-bind the prefix char if it's not currently set.
	# Alpha seems not to bother to bind ctrl-c automatically, for instance.
	if {[regexp {[A-Z]} $prefix]} {
	    if {[bind Alpha <Control-KeyPress-[string tolower $prefix]>] == ""} {
		bind Alpha <Control-KeyPress-[string tolower $prefix]> \
		  "prefixChar Prefix-$prefix ; break"
	    }
	}
    } else {
	append bmode Alpha
	if {$bmode != "Alpha"} {append bmode "Style"}
    }
    #echo [list bind $bmode $bind $script]
    if {$script != ""} { append script " ; break" }
    if {$tcl_platform(platform) == "windows" && [info exists have_option]} {
	set ignore "no meta key"
    }
    if {$key == "Enter"} {
	set ignore "no Enter key"
    }
    if {[info exists ignore]} {
	echo "FYI: keyboard has $ignore; ignoring [list bind $bmode $bind $script]"
	return
    }
    if {[catch [list bind $bmode $bind $script]]} {
	global badkeylog ; lappend badkeylog $bind
    }
}
# need 0x21 0x29 0x1b Enter
array set alpha::menukeymap [list]
array set alpha::keymap [list 0x27 quoteright 0x2f period \
  - minus 0x31 space "\r" Return " " space 0x33 BackSpace \
  "\t" Tab 0x30 Tab "" Left "" Right 0x7b Left 0x7c Right \
  Del Delete Esc Escape 0x7d Down 0x7e Up \
  Pgup Prior Pgdn Next . period , comma \
  "\]" bracketright "\[" bracketleft = equal ? question "/" slash \
  ' quoteright ` quoteleft "\\" backslash ";" semicolon]

# prefixChar - used to further modify the next keystroke 
#  combination, in the same manner as using the shift key 
#  in the next keystroke
proc prefixChar {{bt ""}} {
    if {$bt == ""} {
	alpha::errorAlert "Prefix char called without argument; shouldn't happen!"
    } else {
	message "Prefix..."
	text_cmd binding_capture $bt
    }
}
# startEscape - used to further modify the next 
#  keystroke combination, in the same manner as using the 
#  shift key in the next keystroke
proc startEscape {args} {
    alertnote "Shouldn't call 'startEscape' -- diverted to prefix char!"
}

# float -m <menu> [<-h|-w|-l|-t|-M> <val>] [-n winname] [-z tag] -
#  Takes a created menu (not necessarily in the menubar), and makes a 
#  floating window out of it. Returns integer tag that is used to remove 
#  the window. NOT DYNAMIC!  W/ no options returns all currently defined menus.
#  Displayed w/ system floating window WDEF if system 7.5, plainDBox 
#  otherwise. -h through -M change width, height, left margin, top margin, and
#  margin between buttons. -z allows a ten-char tag to be specified for 
#  'floatShowHide'.
proc float {args} {echo "float $args"}
# floatShowHide <on|off> <tag> - Shows or hides all floats w/ specified 
#  tag. Tags of current modes are automatically shown/hidden.
proc floatShowHide {args} {
    global tearoffMenus
    if {$tearoffMenus} {
	echo "floatShowHide $args"
    }
}
# unfloat <float num> - removes specified floating window. W/ no options 
#  lists all floating windows.
proc unfloat {args} {echo "unfloat $args"}



