/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

/* MENU.C -- Menus (including pulldown bars) with submenus. */

#include <stdlib.h>
#include <graphics.h>
#include <string.h>
#include <assert.h>
#include "key.h"
#include "window.h"
#include "wintern.h"
#include "ctype.h"

enum {
 menu_active_color = EGA_BLACK,
 menu_inactive_color = EGA_LIGHTGRAY,
 menu_border_color = EGA_BLACK,
 menu_bg_color = EGA_WHITE,
 menu_x_margin = 6,
 menu_x_space = 8,
 menu_y_margin = 4,
 menu_button = bLEFT,
 menu_border_width = 2,
 menu_submenu_x_stagger = 8,
};

LOCAL(void) draw_menu_entry(MENU_ENTRY me)
{
# define w (&me->window)
  char buf[16];
  static struct colors {
    char bg, fg;
  } color_tbl[2][2] = {                         /* enab rvs */
    menu_bg_color,      menu_inactive_color,    /* 0    0   */
    menu_bg_color,      menu_inactive_color,    /* 0    1   */
    menu_bg_color,      menu_active_color,      /* 1    0   */
    menu_active_color,  menu_bg_color,          /* 1    1   */
  };
  struct colors colors;

  if (!w_status_p(w, wsVISIBLE))
    return;

/* Silly optimization saves about 5 instructions. */
#if meENABLED == 0
#define not0e
#else
#define not0e 0 !=
#endif
#if meREVERSED == 0
#define not0r
#else
#define not0r 0 !=
#endif

  colors = color_tbl[not0e me_status_p(me, meENABLED)][not0r me_status_p(me, meREVERSED)];
  set_window_bg_color(w, colors.bg, 0);
  push_graphics_state(w, 0);
  setcolor(colors.fg);
  protect_cursor(w);
  out_hot_textxy(menu_x_margin, menu_y_margin, &me->text);

  settextjustify(RIGHT_TEXT, TOP_TEXT);
  outtextxy(w->width - menu_x_margin - 1, menu_y_margin, key2str(me->hot_key, buf));

  if (me_status_p(me, meRULED)) {
    int rule_y = w->height - 1;
    setcolor(menu_active_color);
    setlinestyle(SOLID_LINE, 0, 1);
    line(0, rule_y, w->width - 1, rule_y);
  }
  unprotect_cursor();
  pop_graphics_state();
}
#undef w

/* Mark the current cursor position by reversing
   and drawing it. */
LOCAL(void) mark_menu_cursor(MENU m)
{
  MENU_ENTRY cursor = m->cursor;
  if (cursor != NULL) {
    cursor->status |= bit(meREVERSED);
    draw_menu_entry(cursor);
  }
}

/* Unmark the current cursor position by unreversing
   and drawing it. */
LOCAL(void) unmark_menu_cursor(MENU m)
{
  MENU_ENTRY cursor = m->cursor;
  if (cursor != NULL) {
    cursor->status &= ~bit(meREVERSED);
    draw_menu_entry(cursor);
  }
}

/* Return submenu field of menu entry.  We use
   environment pointer of action closure to save space. */
#define submenu(E)      ((MENU)(E)->action.env)

/* Return non-0 iff a menu cursor entry is a submenu 
   entry and its submenu has been activated (mapped). */
LOCAL(int) submenu_active_p(MENU m)
{
  MENU_ENTRY me = m->cursor;
  return me != NULL && me_status_p(me, meSUBMENU) && w_status_p(&submenu(me)->window, wsVISIBLE);
}

/* Retract any activated submenus of m. */
LOCAL(void) retract(MENU m)
{
  MENU p;

  for (p = m; submenu_active_p(p); p = submenu(p->cursor))
    /* skip */ ;
  while (p != m) {
    unmap_window(&p->window);
    p = p->super;
    unmark_menu_cursor(p);
  }
}

/* Move the cursor of a menu from one
   entry to another with given position. 
   Return non-0 iff the cursor moved. */
LOCAL(int) move_menu_cursor(MENU m, MENU_ENTRY cursor)
{
  if (cursor == m->cursor) {
    mark_menu_cursor(m);
    return 0;
  }
  retract(m);
  unmark_menu_cursor(m);
  m->cursor = cursor;
  mark_menu_cursor(m);
  return 1;
}

/* Function for advancing from a given menu entry to the
   next with wrapping. */
TYPEDEF_LOCAL(MENU_ENTRY) (*DIRECTION)(MENU, MENU_ENTRY);

/* Advance forward to next menu entry. */
LOCAL(MENU_ENTRY) fore(MENU m, MENU_ENTRY me)
{
  return (me == &m->entries[m->n_entries - 1]) ? m->entries : me + 1;
}

/* Advance backward to last menu entry. */
LOCAL(MENU_ENTRY) back(MENU m, MENU_ENTRY me)
{
  return (me == &m->entries[0]) ? &m->entries[m->n_entries - 1] : me - 1;
}

/* Search from given menu entry to find first enabled one in given direction. 
   Give up after looking at all entries and return NULL in that case. */
LOCAL(MENU_ENTRY) enabled_menu_entry(MENU m, MENU_ENTRY me, DIRECTION dir)
{
  int i, n;

  if (me == NULL)
    me = m->entries;
  for (i = 0, n = m->n_entries; i < n && !me_status_p(me, meENABLED); ++i, me = (*dir)(m, me))
    /* skip */;
  return (i < n) ? me : NULL;
}

/* Set a menu so its cursor is either not set
   at all or set to an enabled menu entry. */
LOCAL(void) init_menu_cursor(MENU m, int set_p)
{
  MENU_ENTRY cursor = m->cursor;
  if (cursor != NULL) {
    /* Lazily let cursor in same location of last selection. */
    if (set_p && me_status_p(cursor, meENABLED)) 
      return;
    cursor->status &= ~bit(meREVERSED);
  }
  m->cursor = cursor = set_p ? enabled_menu_entry(m, NULL, fore) : NULL;
  if (cursor != NULL)
    cursor->status |= bit(meREVERSED);
}

/* Retract activated submenus of the root menu of m. */
LOCAL(void) retract_all(MENU m)
{
  /* Find root. */
  while (m->super != NULL) 
    m = m->super;
  retract(m);
  unset_focus(&m->window);
}

/* Execute the action of a given menu entry.  
   Move the cursor to the given entry.
   If the entry is the null entry or not enabled, retract all and return.
   Else if the entry is a submenu entry and the submenu isn't already extended
     extend it.
   Else retract all the menus of the root and do the action. 

   Notes:  
     1. Retract all goes to the root for the case when the action
	is for hotkey of a menu other than the one(s) extended. 
     2. We don't do the extend in an action function because key_p
	is needed to decide how to initialize the key cursor.  It
	keeps the key cursor from appearing while the mouse button
	is down, an esthetic detail. Also, we have to check for the
	submenu flag here anyway to keep from retracting the super. 
	So we might as well do submenu extension at the same time. */
LOCAL(void) do_menu_action(MENU m, MENU_ENTRY me, int key_p)
{
  move_menu_cursor(m, me);
  if (me == NULL || !me_status_p(me, meENABLED))
    retract_all(m);
  else if (me_status_p(me, meSUBMENU)) {
    MENU sm = submenu(me);
    if (!w_status_p(&sm->window, wsVISIBLE)) {
      init_menu_cursor(sm, !me_status_p(me, meACT_ON_PRESS) || key_p);
      map_window(&sm->window);
      set_focus(&sm->window);
    }
  }
  else {
    retract_all(m);
    (*me->action.code)(me->action.env);
  }
}

/* Do the menu action for an entry that is sure not to be NULL. */
LOCAL(void) do_entry_action(MENU_ENTRY me, int key_p)
{
  if (me != NULL)
    do_menu_action((MENU)me->window.parent, me, key_p);
}

/* Called on key or button release to do the action of the
   entry at the current cursor location. */
LOCAL(void) do_cursor_action(MENU m, int key_p)
{              
  if (key_p || !me_status_p(m->cursor, meACT_ON_PRESS))
    do_menu_action(m, m->cursor, key_p);
  else
    retract(m);
}

/* ----- Handlers --------------------------------------------------- */

LOCAL(void) handle_entry_enter_and_press(EVENT e)
{
  MENU_ENTRY me;

  if (generalized_press_p(e, menu_button)) {
    ungrab_mouse();
    me = (MENU_ENTRY)e->mouse.window;
    if (me_status_p(me, meACT_ON_PRESS))
      do_entry_action(me, 0);
    else 
      move_menu_cursor((MENU)me->window.parent, me);
  }            
}

LOCAL(void) handle_entry_depart(EVENT e)
{
  MENU m = (MENU)e->mouse.window->parent;
  if (generalized_release_p(e, menu_button) && !submenu_active_p(m))
    move_menu_cursor(m, NULL);
}

LOCAL(void) handle_entry_hotkey(EVENT e)
{
  MENU_ENTRY me = (MENU_ENTRY)e->hotkey.window;

  if (me_status_p(me, meHOT_HOT_CHAR) && key_hot_p(e->hotkey.key, &me->text)
      || me->hot_key == e->hotkey.key)
    do_entry_action(me, 1);
}

BeginDefDispatch(menu_item)
  DispatchAction(eMAP, draw_menu_entry((MENU_ENTRY)e->map.window))
  Dispatch(eENTRY, handle_entry_enter_and_press)
  Dispatch(eBUTTON_PRESS, handle_entry_enter_and_press)
  Dispatch(eDEPARTURE, handle_entry_depart)
  Dispatch(eHOTKEY, handle_entry_hotkey)
EndDefDispatch(menu_item)

LOCAL(void) handle_menu_map(EVENT e)
{
  MENU m = (MENU)e->map.window,
       super = m->super;
  if (super != NULL) {
    if (super->super == NULL)
      push_root_window(&super->window, 0);
    push_root_window(&m->window, 0);
  }
}

LOCAL(void) handle_menu_unmap(EVENT e)
{
  MENU m = (MENU)e->map.window,
       super = m->super;
  if (super != NULL) { 
    if (super->super == NULL)
      pop_root_window();
    pop_root_window();
  }
}

/* Return new cursor of menu based on key. Don't actually
   set cursor.  Conditionally ignore home and end to 
   eliminate conflicts between the root and leaf menus. */
LOCAL(MENU_ENTRY) menu_cursor_after_key(MENU m, int key, int ignore_home_end_p)
{
  MENU_ENTRY cursor = m->cursor;

  if (cursor != NULL)
    if (m_status_p(m, mBAR))
      switch(key) {

	case LEFT_ARROW:
	  cursor = enabled_menu_entry(m, back(m, cursor), back);
	  break;

	case RIGHT_ARROW:
	  cursor = enabled_menu_entry(m, fore(m, cursor), fore);
	  break;

	case HOME:
	  if (!ignore_home_end_p)
	    cursor = enabled_menu_entry(m, NULL, fore);
	  break;

	case END:
	  if (!ignore_home_end_p)
	    cursor = enabled_menu_entry(m, &m->entries[m->n_entries - 1], back);
	  break;
      }
    else
      switch (key) {

	case UP_ARROW:
	  cursor = enabled_menu_entry(m, back(m, cursor), back);
	  break;

	case DOWN_ARROW:
	  cursor = enabled_menu_entry(m, fore(m, cursor), fore);
	  break;

	case HOME:
	  if (!ignore_home_end_p)
	    cursor = enabled_menu_entry(m, NULL, fore);
	  break;

	case END:
	  if (!ignore_home_end_p)
	    cursor = enabled_menu_entry(m, &m->entries[m->n_entries - 1], back);
	  break;
      }
  return cursor;
}

LOCAL(void) handle_menu_key(EVENT e)
{
  int i, key;
  MENU m = (MENU)e->keystroke.window;
  MENU_ENTRY me;


  key = e->keystroke.key;

  if (key == ESC) {
    /* If at top of menu tree, give up focus, else   
       retract menu and give focus to super-menu. */
    if (m->super == NULL)
      unset_focus(&m->window);
    else {
      unmap_window(&m->window);
      set_focus(&m->super->window);
    }
    return;
  }
  if (key == '\r') {
    do_cursor_action(m, 1);
    return;
  }
  me = menu_cursor_after_key(m, key, 0);
  if (move_menu_cursor(m, me))
    return;

  /* Handle hot char. */
  for (me = m->entries, i = 0; i < m->n_entries; ++me, ++i)
    if (key_match_p(key, &me->text)) {
      do_entry_action(me, 1);
      return;
    }

  /* We didn't handle it. */
  refuse_keystroke(e);
}

LOCAL(void) handle_menu_gain_focus(EVENT e)
{
  MENU m;

  m = (MENU)e->focus.window;
  if (m->super == NULL)
    init_menu_cursor(m, 1);
  mark_menu_cursor(m);
}

LOCAL(void) handle_menu_lose_focus(EVENT e)
{
  MENU m = (MENU)e->focus.window;

  if (!submenu_active_p(m))
    unmark_menu_cursor(m);
}

/* Root menu must respond to hotkey to
   show consecutive submenus. */
LOCAL(void) handle_menu_hotkey(EVENT e)
{
  MENU m = (MENU)e->hotkey.window;

  if (m->super == NULL && submenu_active_p(m)) 
    do_entry_action(menu_cursor_after_key(m, e->hotkey.key, 1), 1);
}

BeginDefDispatch(menu)
  Dispatch(eMAP, handle_menu_map)
  Dispatch(eUNMAP, handle_menu_unmap)
  Dispatch(eKEYSTROKE, handle_menu_key)
  DispatchAction(eBUTTON_PRESS, retract_all((MENU)e->mouse.window))
  DispatchAction(eBUTTON_RELEASE, do_cursor_action((MENU)e->mouse.window, 0))
  Dispatch(eGAIN_FOCUS, handle_menu_gain_focus)
  Dispatch(eLOSE_FOCUS, handle_menu_lose_focus)
  Dispatch(eVISIBLE_HOTKEY, handle_menu_hotkey)
EndDefDispatch(menu)

/* Recursively locate submenus of menu. */
LOCAL(void) do_locate_menu(MENU m, WINDOW parent)
{
  int i, dx, dy; 
  MENU_ENTRY me;

  for (i = 0, me = m->entries; i < m->n_entries; ++i, ++me) {
    if (me_status_p(me, meSUBMENU)) {
      if (m_status_p(m, mBAR)) {
	dx = 0;
	dy = me->window.height + menu_border_width;
      }
      else {
	dx = menu_submenu_x_stagger;
	dy = me->window.height;
      }
      locate_window(&submenu(me)->window, 
		    window_x_wrt(&me->window, parent) + dx, 
		    window_y_wrt(&me->window, parent) + dy, 1);
      do_locate_menu(submenu(me), parent);
    }
  }
}

/* Recursively open menu and its submenus. */
LOCAL(void) do_open_menu(MENU m, WINDOW parent)
{
  int i, x, y, ewd, eht, width, height, bg;
  MENU_ENTRY me;

  /* Entry height. */
  eht = menu_y_margin + textheight("H") + menu_y_margin;

  /* Menu width/height. */
  if (m_status_p(m, mBAR)) {
    width = parent->width;
    height = eht;
    bg = menu_bg_color;
  }
  else {
    width = m->width * textwidth("W");
    height = m->n_entries * eht;
    bg = NO_COLOR;
  }
  /* Open menu window. */
  open_window(&m->window, parent, 0, 0, width, height,
	      menu_border_width, menu_border_color, bg, 
	      bit(eMAP)|bit(eUNMAP)|bit(eKEYSTROKE)|bit(eVISIBLE_HOTKEY)
	     |bit(eBUTTON_PRESS)|bit(eBUTTON_RELEASE)
	     |bit(eGAIN_FOCUS)|bit(eLOSE_FOCUS));
  SetDispatch(&m->window, menu);
  if (m_status_p(m, mSAVE_UNDER))
    set_window_save_under(&m->window); 
  x = y = 0;
  for (i = 0, me = m->entries; i < m->n_entries; ++i, ++me) {

    /* Entry width. */
    ewd = menu_x_margin + textwidth(me->text.str) + menu_x_margin;

    /* Open with entry width for pulldown entry 
       and menu width for menu entry. */
    open_window(&me->window, &m->window, x, y, 
		m_status_p(m, mBAR) ? ewd : width, eht, 
		0, menu_border_color, menu_bg_color, 
		bit(eMAP)|bit(eENTRY)|bit(eDEPARTURE)
	       |bit(eBUTTON_PRESS)|bit(eHOTKEY));
    SetDispatch(&me->window, menu_item);
    map_window(&me->window);

    /* Recursively open subwindows. */
    if (me_status_p(me, meSUBMENU)) {
      submenu(me)->super = m;
      do_open_menu(submenu(me), parent);
    }

    /* Move to next entry location. */
    if (m_status_p(m, mBAR)) 
      x += ewd + menu_x_space;
    else
      y += eht;
  }
}

void open_menu(MENU m, WINDOW parent)
{
  do_open_menu(m, parent);
  do_locate_menu(m, parent);
}
