// SPDX-License-Identifier: BSD-3-Clause
package org.islandoftex.albatross

import java.io.IOException
import java.nio.file.Paths

object FCList {
    private val descriptions = mapOf(
        "ttf" to "TrueType Font",
        "otf" to "OpenType Font",
        "ttc" to "TrueType Collection"
    )

    @Throws(IOException::class)
    private fun getCommandOutput(command: List<String>): List<String> {
        return ProcessBuilder(command).start().inputStream
            .bufferedReader().useLines { it.toList() }
    }

    @JvmStatic
    val version: String
        @Throws(IOException::class)
        get() = getCommandOutput(listOf("fc-list", "--help"))[0]

    @JvmStatic
    val isAvailable: Boolean by lazy {
        runCatching { version.isNotBlank() }.getOrElse { false }
    }

    @Throws(IOException::class)
    private fun getFontsFor(charset: String): List<String> =
        getCommandOutput(listOf("fc-list", ":charset=$charset"))

    private fun parse(output: List<String>) =
        output.takeIf { it.isNotEmpty() && !output[0].trim().startsWith("Unable") }
            ?.map { line ->
                val (path, spec) = if (line.startsWith("/")) line.split(":", limit = 2).map { it.trim() }
                else line.split(":", limit = 3).map { it.trim() }.let { (p, q, r) ->
                    listOf(p + ":" + q, r)
                }
                val name = spec.substringBefore(":")
                val styles = if (spec.contains(":")) spec.substringAfter(":").trim() else ""
                FontEntry(
                    path = Paths.get(path),
                    name = name,
                    styles = styles.substringAfter("=").split(",").map { it.trim() },
                    description = descriptions[path.substringAfterLast(".").toLowerCase()] ?: "Unknown"
                )
            }?.sortedBy { it.name.toLowerCase() } ?: emptyList()

    @JvmStatic
    @Throws(IOException::class)
    fun getFonts(charset: String): List<FontEntry> = parse(getFontsFor(charset))
}
